/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow;

import com.metaeffekt.artifact.terms.model.LicenseTextProvider;
import com.metaeffekt.artifact.terms.model.NormalizationMetaData;
import com.metaeffekt.resource.InventoryResource;
import lombok.Getter;
import org.metaeffekt.core.inventory.processor.model.Inventory;

import java.io.File;

public class ScanFlowParam extends ConfiguredFlowParam<ScanFlowParam> {

    @Getter
    private NormalizationMetaData normalizationMetaData;

    @Getter
    private LicenseTextProvider licenseTextProvider;

    @Getter
    private File analysisBaseDir;

    @Getter
    private InventoryResource inventoryResource;

    @Getter
    private File resultTargetDir;

    private ScanFlowParam() {
    }

    /**
     * The copy constructor was introduced to support templating {@link ScanFlowParam}.
     *
     * @param scanFlowParam The {@link ScanFlowParam} to copy.
     * @return A copy of the given {@link ScanFlowParam}.
     */
    public static ScanFlowParam copy(ScanFlowParam scanFlowParam) {
        ScanFlowParam sfp = new ScanFlowParam();
        sfp.withInventory(scanFlowParam.inventoryResource);
        sfp.using(scanFlowParam.analysisBaseDir);
        sfp.using(scanFlowParam.normalizationMetaData);
        sfp.using(scanFlowParam.licenseTextProvider);
        sfp.configuredBy(scanFlowParam.getPropertyProvider());
        return sfp;
    }

    public static ScanFlowParam build() {
        return new ScanFlowParam();
    }

    public File getInventoryFile() {
        return inventoryResource.getFile();
    }

    public ScanFlowParam using(NormalizationMetaData normalizationMetaData) {
        this.normalizationMetaData = normalizationMetaData;
        return this;
    }

    public ScanFlowParam using(LicenseTextProvider licenseTextProvider) {
        this.licenseTextProvider = licenseTextProvider;
        return this;
    }

    public ScanFlowParam withInventory(Inventory inventory) {
        this.inventoryResource = InventoryResource.fromInventory(inventory);
        return this;
    }

    public ScanFlowParam withInventory(InventoryResource inventoryResource) {
        this.inventoryResource = inventoryResource;
        return this;
    }

    public ScanFlowParam using(File analysisBaseDir) {
        this.analysisBaseDir = analysisBaseDir;
        return this;
    }

    public ScanFlowParam publishResultsTo(File resultTargetDir) {
        this.resultTargetDir = resultTargetDir;
        return this;
    }

    public File getResultTargetDir() {
        return resultTargetDir;
    }

    public int getScanThreads() {
        return Integer.parseInt(getPropertyProvider().getProperty("workbench.scanner.threads", "4"));
    }
}
