/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng;

import com.metaeffekt.artifact.analysis.flow.ng.keyholder.UserKeysForConsumer;
import com.metaeffekt.artifact.analysis.flow.ng.keyholder.UserKeysForSupplier;
import com.metaeffekt.artifact.analysis.flow.ng.keyholder.UserKeysStorage;
import com.metaeffekt.artifact.terms.model.NormalizationMetaData;
import com.metaeffekt.artifact.terms.model.TermsMetaData;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

public class EncryptedArtifactProducerUtils {

    /**
     * Simpler replacement for junit's assertEquals.
     * <br>
     * Used since this code now doesn't run under junit any more (for runtime tests).
     *
     * @param o1 an object to compare to the second
     * @param o2 the second object to be compared
     */
    public static void assertEquals(Object o1, Object o2) {
        if (!Objects.equals(o1, o2)) {
            throw new RuntimeException("Self-test failed. " + o1 + " does not equal " + o2);
        }
    }

    public static void assertWordlistEquals(Collection<String> actualWordlist, Collection<String> expectedWordlist) {
        // REVIEW-JKR: out of nowhere the lists must be ordered to be equal; investigate;
        //   maybe words have the same occurrence count and are therefore not deteministically ordered

        final List<String> readWordlistArray = actualWordlist.stream().sorted().collect(Collectors.toList());
        final List<String> generatedWordlistArray = expectedWordlist.stream().sorted().collect(Collectors.toList());

        for (int i = 0; i < actualWordlist.size(); i++) {
            assertEquals(readWordlistArray.get(i), generatedWordlistArray.get(i));
        }
    }

    private static void mkdirOrErr(File... dirs) throws IOException {
        for (File toMkdir : dirs) {
            mkdirOrErr(toMkdir);
        }
    }

    /**
     * Utility method to write a dummy keypair for testing.
     *
     * @param dummyKeyForSupplierFile where the dummy key (part for suppliers) will be written to
     * @param dummyKeyForConsumerFile where the dummy key (part for dummy consumers) will be written to
     * @param keyForSupplier the key object to be written
     * @param keyForConsumer the key object to be written
     * @param temporaryPassword password to use for dummy key encryption (one should delete it after the test is done)
     *
     * @throws IOException throws on failure
     */
    public static void writeDummyKeys(File dummyKeyForSupplierFile,
                                      File dummyKeyForConsumerFile,
                                      UserKeysForSupplier keyForSupplier,
                                      UserKeysForConsumer keyForConsumer,
                                      String temporaryPassword) throws IOException {
        // generate an additional authorized keyslot
        dummyKeyForSupplierFile.deleteOnExit();
        dummyKeyForConsumerFile.deleteOnExit();
        mkdirOrErr(dummyKeyForSupplierFile.getParentFile(), dummyKeyForConsumerFile.getParentFile());
        UserKeysStorage.writeUserKeys(keyForSupplier, dummyKeyForSupplierFile);
        UserKeysStorage.writeUserKeys(keyForConsumer, temporaryPassword, dummyKeyForConsumerFile);
    }

    /**
     * Run mkdir for the given file('s path).<br>
     * Throw if
     * <ul>
     *     <li>the directory does not exist and can't be created</li>
     *     <li>the given file exists and is not a directory.</li>
     * </ul>
     *
     * @param toMkdir the location to ensure a directory is at.
     * @throws IOException throws on failure to ensure a directory exists in the specified location.
     */
    public static void mkdirOrErr(File toMkdir) throws IOException {
        if (!toMkdir.exists()) {
            if (!toMkdir.mkdir()) {
                throw new IOException("Couldn't create '" + toMkdir.getPath() + "'");
            }
        }
        if (!toMkdir.isDirectory()) {
            throw new IOException("Should-be directory '" + toMkdir.getPath() + "' isn't a directory");
        }
    }

    /**
     * Filters metadata from the customer directory.<br>
     * Useful for preparing normalizationMetaData before producing encrypted artifacts using said normMeta.
     * @param normalizationMetaData the object to be modified. (acts as in-out of this method)
     */
    public static void filterCustomerMetaData(NormalizationMetaData normalizationMetaData) {
        for (TermsMetaData tmd : new HashSet<>(normalizationMetaData.getLicenseMetaDataMap().values())) {
            if (tmd.isCustomerMetaData()) {
                normalizationMetaData.remove(tmd);
            }
        }
    }

}
