/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng;

import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * Collection of static methods useful for reading stuff.
 */
public class StreamReadingUtils {
    /**
     * Tries to read a part of a stream and throws if it can't for whatever reason.<br>
     * The size of the buffer to read to determines how many bytes are read.
     * @param inputStream the stream to read from
     * @param toReadTo the byte array to read to
     * @return returns how many bytes were read from the stream after summation of the amount read with each read call
     * @throws IOException throws on failure to read
     */
    public static int readOrThrow(InputStream inputStream, byte[] toReadTo) throws IOException {
        int readPos = 0;
        while (readPos < toReadTo.length) {
            int read = inputStream.read(toReadTo, readPos, toReadTo.length - readPos);
            if (read >= 0) {
                // we either succeeded in reading or read nothing.
                readPos += read;
            } else {
                // read failure. throw hard.
                throw new IOException("Could not read iv from content file.");
            }
        }

        if (readPos > toReadTo.length) {
            throw new RuntimeException("Sanity check failed: overrun while reading bytes");
        }

        return readPos;
    }

    /**
     * Attempts to find an entry with the given name in the zip file.
     * @param zipInputStream needs to be a fresh ZipInputStream to be able to search all entries
     * @param entryName the name of entry to be searched for
     * @return returs true on success, false on failure to find the entry.
     * @throws IOException throws IOException if the zip file could not be read properly
     */
    public static boolean skipToEntry(ZipInputStream zipInputStream, String entryName) throws IOException {
        ZipEntry entry = zipInputStream.getNextEntry();
        while (entry != null) {

            // find the entry we need
            if (entry.getName().equals(entryName)) {
                // success
                return true;
            }

            entry = zipInputStream.getNextEntry();
        }

        // could not find this entry
        return false;
    }
}
