/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.crypt;

import java.security.SecureRandom;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Simple keyholder for convenience
 */
public final class ContentEncryptionKey {
    private final AtomicBoolean burned = new AtomicBoolean(false);
    private byte[] rawContentEncryptionKey;

    // use 16 bytes for AES-128, use 32 byte key to use AES-256
    // TODO: consider updating to use aes-256 once we use java>8 where unlimited security policies are the default!
    private static final int length = 16;

    public ContentEncryptionKey() {
        final SecureRandom secureRandom = new SecureRandom();
        this.rawContentEncryptionKey = new byte[length];
        secureRandom.nextBytes(rawContentEncryptionKey);
    }


    public synchronized byte[] getRaw() {
        if (burned.get()) {
            throw new IllegalStateException("Tried to use object after burning it.");
        }
        return rawContentEncryptionKey.clone();
    }

    /**
     * Gets the length of the contained content encryption key.
     * @return the length of the held key
     */
    public int getLength() {
        if (burned.get()) {
            throw new IllegalStateException("Tried to use object after burning it.");
        }
        return length;
    }

    public synchronized void burn() {
        burned.set(true);
        rawContentEncryptionKey = null;
    }
}
