/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.crypt.licensetexts;

import com.fasterxml.jackson.databind.MappingIterator;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.metaeffekt.artifact.analysis.flow.ng.crypt.EncryptedZipProvider;
import com.metaeffekt.artifact.analysis.flow.ng.crypt.param.ProviderParameters;
import com.metaeffekt.artifact.terms.model.LicenseTextEntry;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;

import static com.metaeffekt.artifact.analysis.flow.ng.EncryptedArchiveConstants.LICENSE_TEXT_DB_ENTRY_NAME;

public class EncryptedTextCollectionReader {

    /**
     * Tries to find license texts for the given canonical names in a license text database.
     * @param param used in decryption
     * @param canonicalNameSet the names to look for
     * @return returns a list of found pairs of the form canonicalName to {@link LicenseTextEntry}.
     */
    public Map<String, LicenseTextEntry> findLicenses(ProviderParameters param, Set<String> canonicalNameSet) {
        final EncryptedZipProvider provider = new EncryptedZipProvider(param);

        final Map<String, LicenseTextEntry> licenseTextEntries = new HashMap<>();

        try (InputStream inputStream = provider.getEntryStream(LICENSE_TEXT_DB_ENTRY_NAME);
            final GZIPInputStream gzipInputStream = new GZIPInputStream(inputStream);
            final InputStreamReader reader = new InputStreamReader(gzipInputStream, StandardCharsets.UTF_8)) {
            final ObjectReader licenseTextEntryReader = new ObjectMapper().readerFor(LicenseTextEntry.class);
            final MappingIterator<Object> it = licenseTextEntryReader.readValues(reader);

            try {
                // simple linear search so we don't ever have to load all data to memory
                while (it.hasNextValue()) {
                    final Object o = it.nextValue();
                    // something is very wrong if this cast fails (db version?)
                    final LicenseTextEntry licenseTextEntry = (LicenseTextEntry) o;

                    if (canonicalNameSet.contains(licenseTextEntry.getCanonicalName())) {
                        licenseTextEntries.put(licenseTextEntry.getCanonicalName(), licenseTextEntry);
                    }
                }
            } catch (IOException e) {
                throw new IllegalStateException("Unreadable json while parsing license text entries.", e);
            }
        } catch (Exception e) {
            throw new RuntimeException(
                    "An exception likely indicates misconfiguration (for example): " +
                        "dataset / scanner version mismatch, " +
                        "keys unauthorized for this artifact, key or data integrity error, " +
                        "invalid dataset or keyfile locations, (or other). " +
                        "Please contact {metaeffekt} for support.", e
            );
        }
        return licenseTextEntries;
    }
}
