/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.crypt.licensetexts;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SequenceWriter;
import com.metaeffekt.artifact.analysis.flow.ng.crypt.EncryptedEntryOutputStream;
import com.metaeffekt.artifact.analysis.flow.ng.crypt.EncryptedZipSupplier;
import com.metaeffekt.artifact.analysis.flow.ng.crypt.param.NormMetaSupplierParameters;
import com.metaeffekt.artifact.terms.model.LicenseTextEntry;
import com.metaeffekt.artifact.terms.model.NormalizationMetaData;
import com.metaeffekt.artifact.terms.model.ResolverUtils;
import com.metaeffekt.artifact.terms.model.TermsMetaData;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.NoSuchPaddingException;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.util.Map;
import java.util.zip.GZIPOutputStream;

import static com.metaeffekt.artifact.analysis.flow.ng.EncryptedArchiveConstants.LICENSE_TEXT_DB_ENTRY_NAME;

/**
 * Implements writing a collection of license texts (indexed by canonical name).<br>
 * Use {@link EncryptedTextCollectionReader} for reading.
 */
public class EncryptedTextCollectionSupplier {
    private static final Logger LOG = LoggerFactory.getLogger(EncryptedTextCollectionSupplier.class);

    public void process(NormMetaSupplierParameters param)
            throws NoSuchPaddingException, InvalidAlgorithmParameterException, NoSuchAlgorithmException, IOException,
            NoSuchProviderException, InvalidKeyException {
        writeData(param);
    }

    protected void writeData(NormMetaSupplierParameters param)
            throws IOException, NoSuchProviderException, NoSuchPaddingException, NoSuchAlgorithmException,
            InvalidKeyException, InvalidAlgorithmParameterException {
        try (final EncryptedZipSupplier supplier = new EncryptedZipSupplier(param)) {
            try (final EncryptedEntryOutputStream entryOutputStream =
                         supplier.getEncryptedEntryStream(LICENSE_TEXT_DB_ENTRY_NAME);
                 final GZIPOutputStream gzipOutputStream = new GZIPOutputStream(entryOutputStream, 65536);
                 final OutputStreamWriter writer = new OutputStreamWriter(gzipOutputStream, StandardCharsets.UTF_8);
                 final SequenceWriter seq = new ObjectMapper()
                         .writerFor(LicenseTextEntry.class)
                         .withRootValueSeparator("\n")
                         .writeValues(writer)) {
                NormalizationMetaData normMeta = param.getNormalizationMetaData();

                // prepare and write all license files, keeping their: filename, content, canonical name
                for (Map.Entry<String, TermsMetaData> e : normMeta.getLicenseMetaDataMap().entrySet()) {
                    TermsMetaData tmd = e.getValue();

                    File foundLicenseFile = ResolverUtils.findLicenseFor(tmd, normMeta);

                    if (foundLicenseFile == null) {
                        LOG.trace("No valid license file for license [{}].", tmd.getCanonicalName());
                        continue;
                    }

                    String licenseText;
                    try (InputStream inputStream = Files.newInputStream(foundLicenseFile.toPath());
                         Reader reader = new InputStreamReader(inputStream, StandardCharsets.UTF_8)) {
                        licenseText = IOUtils.toString(reader);
                    }

                    LicenseTextEntry licenseEntry = new LicenseTextEntry(
                            tmd.getCanonicalName(),
                            foundLicenseFile.getName(),
                            licenseText
                    );
                    seq.write(licenseEntry);
                }
            }
        }
    }
}
