/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.crypt.param;

import com.metaeffekt.artifact.analysis.flow.ng.ContentAlgorithmParam;

import java.io.File;
import java.util.Objects;

/**
 * Class with basic parameters needed for {@link com.metaeffekt.artifact.analysis.flow.ng.crypt.EncryptedZipSupplier}.
 */
public class SupplierParameters {
    /**
     * Parameters about the algorithms used in encryption.
     */
    private final ContentAlgorithmParam contentAlgorithmParam;

    /**
     * A directory containing {@link com.metaeffekt.artifact.analysis.flow.ng.keyholder.UserKeysForSupplier} files.
     */
    private final File allowedUserKeysDir;

    /**
     * A file containing license information for unencrypted inclusion in the archive.
     */
    private final File licenseTextFile;

    /**
     * This is where the output zip will be written to. It simpy contains the keyslots file and encrypted object.
     */
    private final File zipOutputFile;

    /**
     * This many keyslots will be generated. Also serves as the maximum amount of usable keyslots in a release.
     * <br>
     * This value should ONLY ever increase (ideally) by doubling it and SHOULD NEVER DECREASE AGAIN except in extreme cases.
     * <br>
     * The number of keyslots must always be this value.
     * Unused keyslots must be padded with bogus slots indistinguishable from actual used keyslots.
     * <br>
     * Correct implementation should guarantee privacy for the number of users that have access (keys) to the artifact.
     */
    public final int keyslotNum = 256;

    public SupplierParameters(
            ContentAlgorithmParam contentAlgorithmParam,
            File allowedUserKeysDir,
            File licenseTextFile,
            File zipOutputFile) {
        this.contentAlgorithmParam = Objects.requireNonNull(contentAlgorithmParam);
        this.allowedUserKeysDir = Objects.requireNonNull(allowedUserKeysDir);
        this.licenseTextFile = Objects.requireNonNull(licenseTextFile);
        this.zipOutputFile = Objects.requireNonNull(zipOutputFile);
    }

    public ContentAlgorithmParam getContentAlgorithmParam() {
        return contentAlgorithmParam;
    }

    public File getAllowedUserKeysDir() {
        return allowedUserKeysDir;
    }

    public File getLicenseTextFile() {
        return licenseTextFile;
    }

    public File getZipOutputFile() {
        return zipOutputFile;
    }
}
