/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.crypt.tmd;

import com.metaeffekt.artifact.analysis.flow.ng.crypt.EncryptedEntryOutputStream;
import com.metaeffekt.artifact.analysis.flow.ng.crypt.EncryptedZipSupplier;
import com.metaeffekt.artifact.analysis.flow.ng.crypt.param.NormMetaSupplierParameters;
import com.metaeffekt.flow.common.AbstractFlow;

import javax.crypto.NoSuchPaddingException;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.util.zip.GZIPOutputStream;

import static com.metaeffekt.artifact.analysis.flow.ng.EncryptedArchiveConstants.DATASET_ENTRY_NAME;

/**
 * Use this through TermMetaDataProducer to make sure all necessary steps for packaging are performed.
 */
public class EncryptedTermMetaDataSupplierNG extends AbstractFlow {
    public void process(NormMetaSupplierParameters param)
            throws NoSuchPaddingException, InvalidAlgorithmParameterException, NoSuchAlgorithmException, IOException,
            NoSuchProviderException, InvalidKeyException {
        try (final EncryptedZipSupplier zipSupplier = new EncryptedZipSupplier(param)) {
            writeTermMetaData(zipSupplier, param);
        }
    }

    protected void writeTermMetaData(EncryptedZipSupplier zipSupplier, NormMetaSupplierParameters param)
        throws IOException, InvalidAlgorithmParameterException, NoSuchPaddingException, NoSuchAlgorithmException,
        NoSuchProviderException, InvalidKeyException {
        // write content as an encrypted object
        try (final EncryptedEntryOutputStream outputStream = zipSupplier.getEncryptedEntryStream(DATASET_ENTRY_NAME);
             final GZIPOutputStream gzipOutputStream = new GZIPOutputStream(outputStream, 65536);
             final ObjectOutputStream objectOutputStream = new ObjectOutputStream(gzipOutputStream)) {
            objectOutputStream.writeObject(param.getNormalizationMetaData());
        }
    }
}
