/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.keyholder;

import org.bouncycastle.crypto.params.RSAKeyParameters;

/**
 * Holder object containing data required for decryption of a content object.
 */
public class UserKeysForConsumer implements UserKeysWithHmacSecret {
    private final RSAKeyParameters rsaPrivateKeyParameters;
    private final byte[] hmacSecretKey;

    /**
     * Constructs a new object holding the provided parameters
     * @param rsaPrivateKeyParams is the public key in RsaKeyParameters form, used for decrypting key encapsulation.
     * @param hmacSecretKey is a secret key for hmac such as in finding one's correct keyslot.
     */
    public UserKeysForConsumer(RSAKeyParameters rsaPrivateKeyParams, byte[] hmacSecretKey) {
        if (rsaPrivateKeyParams == null) {
            throw new IllegalArgumentException("private key may not be null.");
        }
        if (!rsaPrivateKeyParams.isPrivate()) {
            throw new IllegalArgumentException("Expecting a private key for the consumer.");
        }
        if (hmacSecretKey == null) {
            throw new IllegalArgumentException("hmacSecretKey may not be null.");
        }

        this.rsaPrivateKeyParameters = rsaPrivateKeyParams;
        this.hmacSecretKey = hmacSecretKey;
    }

    public RSAKeyParameters getRsaPrivateKey() {
        return rsaPrivateKeyParameters;
    }

    public byte[] getHmacSecretKey() {
        return hmacSecretKey.clone();
    }
}
