/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.keyholder;

import org.bouncycastle.crypto.params.RSAKeyParameters;

/**
 * Handy holder object containing all info the supplier (of an encrypted artifact) needs for generation.
 */
public class UserKeysForSupplier implements UserKeysWithHmacSecret {
    private final RSAKeyParameters rsaPublicKeyParameters;
    private final byte[] hmacSecretKey;

    /**
     * Constructs a new object holding the provided parameters
     * @param rsaPublicKeyParameters is the public key in RsaKeyParameters form, used for key encapsulation.
     * @param hmacSecretKey is a secret key for later use in hmac generation.
     */
    public UserKeysForSupplier(RSAKeyParameters rsaPublicKeyParameters, byte[] hmacSecretKey) {
        if (rsaPublicKeyParameters == null) {
            throw new IllegalArgumentException("public key may not be null.");
        }
        if (rsaPublicKeyParameters.isPrivate()) {
            throw new IllegalArgumentException("Expecting a public key for the supplier.");
        }
        if (hmacSecretKey == null) {
            throw new IllegalArgumentException("hmacSecretKey may not be null.");
        }

        this.rsaPublicKeyParameters = rsaPublicKeyParameters;
        this.hmacSecretKey = hmacSecretKey;
    }

    public RSAKeyParameters getRsaPublicKey() {
        return rsaPublicKeyParameters;
    }

    public byte[] getHmacSecretKey() {
        return hmacSecretKey.clone();
    }
}
