/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.ng.keyholder;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapType;
import com.metaeffekt.artifact.analysis.flow.ng.StreamReadingUtils;
import com.metaeffekt.artifact.analysis.flow.ng.exception.SelfcheckFailedException;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.pkcs.RSAPublicKey;
import org.bouncycastle.crypto.DerivationFunction;
import org.bouncycastle.crypto.DerivationParameters;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.generators.HKDFBytesGenerator;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.HKDFParameters;
import org.bouncycastle.crypto.params.RSAKeyParameters;
import org.bouncycastle.crypto.util.PrivateKeyFactory;
import org.bouncycastle.crypto.util.PrivateKeyInfoFactory;

import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.security.*;
import java.util.Base64;
import java.util.TreeMap;

public class UserKeysStorage {

    /**
     * Encryption cipher for consumer keyfiles. Changing this will break compatibility with all existing keyfiles.
     * @param key the used cryptographic key
     * @param iv the used cryptographic (unique) iv
     * @return returns the initialized cipher, ready for use
     * @throws NoSuchPaddingException throws on failure
     * @throws NoSuchAlgorithmException throws on failure
     * @throws NoSuchProviderException throws on failure
     * @throws InvalidAlgorithmParameterException throws on failure
     * @throws InvalidKeyException throws on failure
     */
    public static Cipher getConsumerKeyEncryptionCipher(Key key, byte[] iv)
            throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException,
            InvalidAlgorithmParameterException, InvalidKeyException {
        Cipher cipher = Cipher.getInstance("AES" + "/EAX/NoPadding", "BC");
        if (iv == null) {
            throw new IllegalArgumentException("IV should not be null with this scheme.");
        }

        GCMParameterSpec spec = new GCMParameterSpec(128, iv);
        cipher.init(Cipher.ENCRYPT_MODE, key, spec);
        return cipher;
    }

    /**
     * Decryption cipher for consumer keyfiles. Changing this will break compatibility with all existing keyfiles.
     * @param key the used cryptographic key
     * @param iv the used cryptographic (unique) iv
     * @return returns the initialized cipher, ready for use
     * @throws NoSuchPaddingException throws on failure
     * @throws NoSuchAlgorithmException throws on failure
     * @throws NoSuchProviderException throws on failure
     * @throws InvalidAlgorithmParameterException throws on failure
     * @throws InvalidKeyException throws on failure
     */
    public static Cipher getConsumerKeyDecryptionCipher(Key key, byte[] iv)
            throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException,
            InvalidAlgorithmParameterException, InvalidKeyException {
        Cipher cipher = Cipher.getInstance("AES" + "/EAX/NoPadding", "BC");
        if (iv == null) {
            throw new IllegalArgumentException("IV should not be null with this scheme.");
        }

        GCMParameterSpec spec = new GCMParameterSpec(128, iv);
        cipher.init(Cipher.DECRYPT_MODE, key, spec);
        return cipher;
    }

    /**
     * Transforms the user password into a key for keyfile decryption.<br>
     * Outputs a secret key suitable for use with AES.
     * @param password the password to be turned into a secret key
     * @param untrustedSalt salt (may be untrusted) to use in turning the password into a secret key
     * @return a ready-to-use key with a length of 16 bytes for keyfile decryption.
     */
    public static Key passwordTo16ByteKey(String password, byte[] untrustedSalt) {
        // note that untrusted salts like this go into the "info" field in hkdf. it's just kind of weird that way.
        DerivationParameters param = new HKDFParameters(password.getBytes(StandardCharsets.UTF_8), null, untrustedSalt);
        DerivationFunction gen = new HKDFBytesGenerator(new SHA256Digest());
        gen.init(param);

        byte[] output = new byte[16];
        // since we get an otherwise useless return value here, use it for a sanity check
        if (gen.generateBytes(output, 0, 16) != 16) {
            throw new RuntimeException("Sanity check failed: HKDF didn't claim to have written the correct 16 bytes.");
        }

        return new SecretKeySpec(output, "AES");
    }

    /**
     * Get a selfcheck sum derived from two Strings. Should pass the encoded asymmetric key and secret in that order.
     * @param encodedAsymmetric the string (as it appears in the keyfile)
     * @param encodedSecret an encoded secret (as it appears in the keyfile)
     * @return the string containing the base64 encoded (self-)checksum
     */
    public static String getSelfcheckSum(String encodedAsymmetric, String encodedSecret) {
        MessageDigest digest;

        try {
            digest = MessageDigest.getInstance("SHA-256");
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }

        digest.update(encodedAsymmetric.getBytes(StandardCharsets.UTF_8));
        digest.update(encodedSecret.getBytes(StandardCharsets.UTF_8));

        return Base64.getEncoder().encodeToString(digest.digest());
    }

    /**
     * Writes the input object to a custom file.
     * @param userKeysForSupplier what will be written
     * @param outputFile where to write the file
     * @throws IOException throws on failure to write to the file
     */
    public static void writeUserKeys(UserKeysForSupplier userKeysForSupplier, File outputFile) throws IOException {
        TreeMap<String, String> interMap = toInterMap(userKeysForSupplier);

        ObjectMapper mapper = new ObjectMapper();
        try (OutputStream fos = Files.newOutputStream(outputFile.toPath(),
                StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING)) {
            // write output. jackson docs say it uses UTF-8 by default so we'll let it handle the writer.
            mapper.writerWithDefaultPrettyPrinter().writeValue(fos, interMap);
        }
    }

    /**
     * Writes the input object to a custom file
     * @param userKeysForConsumer what will be written
     * @param password password to encrypt the consumer keyfile with
     * @param outputFile where to write the file
     * @throws IOException throws on failure to write to the file
     */
    public static void writeUserKeys(UserKeysForConsumer userKeysForConsumer, String password, File outputFile) throws IOException {
        TreeMap<String, String> interMap = toInterMap(userKeysForConsumer);

        byte[] iv = new byte[16];
        new SecureRandom().nextBytes(iv);

        // reuse the iv as an untrusted kdf salt
        Key derivedEncryptionKey = passwordTo16ByteKey(password, iv);

        try (OutputStream fos = Files.newOutputStream(outputFile.toPath(),
                StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE)) {
            fos.write(iv);

            Cipher cipher = getConsumerKeyEncryptionCipher(derivedEncryptionKey, iv);
            ObjectMapper mapper = new ObjectMapper();
            try (CipherOutputStream cipherOutputStream = new CipherOutputStream(fos, cipher)) {
                // write output. jackson docs say it uses UTF-8 by default so we'll let it handle the writer.
                mapper.writerWithDefaultPrettyPrinter().writeValue(cipherOutputStream, interMap);
            }
        } catch (InvalidAlgorithmParameterException | NoSuchPaddingException | NoSuchAlgorithmException |
                 NoSuchProviderException | InvalidKeyException e) {
            throw new RuntimeException(e);
        }
    }

    public static UserKeysForSupplier readUserKeysForPublisher(File inputFile, boolean doSelfcheck) throws IOException, SelfcheckFailedException {
        if (!inputFile.isFile()) {
            throw new IOException("Input should be a file.");
        }

        TreeMap<String, String> interMap;

        ObjectMapper mapper = new ObjectMapper();
        // unfortunately, jackson doesn't specify the encoding it uses for reading. we need our own reader.
        try (InputStream fis = Files.newInputStream(inputFile.toPath());
             InputStreamReader inputReader = new InputStreamReader(fis, StandardCharsets.UTF_8)) {
            MapType stringMap = mapper.getTypeFactory().constructMapType(TreeMap.class, String.class, String.class);
            interMap = mapper.readValue(inputReader, stringMap);
        }

        if (interMap == null) {
            throw new RuntimeException("Should not happen: interMap was null after reading or throwing.");
        }

        return fromInterMapForPublisher(interMap, doSelfcheck);
    }

    public static UserKeysForConsumer readUserKeysForConsumer(File inputFile, String password, boolean doSelfcheck)
            throws IOException, SelfcheckFailedException {
        if (!inputFile.isFile()) {
            throw new IOException("Input should be a file. got " + inputFile.getAbsolutePath());
        }

        String jsonString;

        // unfortunately, jackson doesn't specify the encoding it uses for reading. we need our own reader.
        try (InputStream fis = Files.newInputStream(inputFile.toPath(), StandardOpenOption.READ)) {
            // read iv first
            byte[] iv = new byte[16];
            StreamReadingUtils.readOrThrow(fis, iv);

            Key key = passwordTo16ByteKey(password, iv);
            SecretKeySpec secretKeySpec = (SecretKeySpec) key;

            Cipher decryptionCipher = getConsumerKeyDecryptionCipher(key, iv);

            // FIXME: read the entire file to memory first so MAC check fails before we begin parsing
            try (CipherInputStream cipherInputStream = new CipherInputStream(fis, decryptionCipher);
                    InputStreamReader inputReader = new InputStreamReader(cipherInputStream, StandardCharsets.UTF_8)) {
                jsonString = IOUtils.toString(inputReader);
            }
        } catch (InvalidAlgorithmParameterException | NoSuchPaddingException | NoSuchAlgorithmException |
                 NoSuchProviderException | InvalidKeyException e) {
            throw new RuntimeException(e);
        }

        ObjectMapper mapper = new ObjectMapper();
        MapType stringMap = mapper.getTypeFactory().constructMapType(TreeMap.class, String.class, String.class);
        TreeMap<String, String> interMap = mapper.readValue(jsonString, stringMap);

        if (interMap == null) {
            throw new RuntimeException("Should not happen: interMap was null after reading or throwing.");
        }

        return fromInterMapForUser(interMap, doSelfcheck);
    }

    public static UserKeysForConsumer fromInterMapForUser(TreeMap<String, String> interMap, boolean doSelfcheck)
            throws IOException, SelfcheckFailedException {
        String base64Pkcs8EncodedPrivateKey = interMap.get("rsaPrivateKey");
        String encodedHmacKey = interMap.get("secretHmacKey");

        String receivedSelfcheck = interMap.get("selfcheckSum");
        if (doSelfcheck && !receivedSelfcheck.equals(getSelfcheckSum(base64Pkcs8EncodedPrivateKey, encodedHmacKey))) {
            throw new SelfcheckFailedException("Keyfile's self-check failed: Keyfile is corrupt and might not work!");
        }

        Base64.Decoder decoder = Base64.getDecoder();

        AsymmetricKeyParameter privateKey = PrivateKeyFactory.createKey(decoder.decode(base64Pkcs8EncodedPrivateKey));

        return new UserKeysForConsumer((RSAKeyParameters) privateKey, decoder.decode(encodedHmacKey));
    }

    public static UserKeysForSupplier fromInterMapForPublisher(TreeMap<String, String> interMap, boolean doSelfcheck)
            throws IOException, SelfcheckFailedException {
        String base64DerEncodedPublicKey = interMap.get("rsaPublicKey");
        String encodedHmacKey = interMap.get("secretHmacKey");

        // serious error if those keys are null or not present
        if (base64DerEncodedPublicKey == null) {
            throw new RuntimeException("rsaPublicKey from json can't be null");
        }
        if (encodedHmacKey == null) {
            throw new RuntimeException("secretHmacKey from json can't be null");
        }

        String receivedSelfcheck = interMap.get("selfcheckSum");
        if (doSelfcheck && !receivedSelfcheck.equals(getSelfcheckSum(base64DerEncodedPublicKey, encodedHmacKey))) {
            throw new SelfcheckFailedException("Keyfile's self-check failed: Keyfile is corrupt and might not work!");
        }

        // decode base64
        Base64.Decoder decoder = Base64.getDecoder();
        byte[] derEncodedPublicKey = decoder.decode(base64DerEncodedPublicKey);

        // clumsy syntax to read a DER RsaPublicKey back to our java object: first get the primitive, then the object.
        ASN1Primitive derPublicKeyPrimitive = ASN1Primitive.fromByteArray(derEncodedPublicKey);
        RSAPublicKey rsaPublicKey = RSAPublicKey.getInstance(derPublicKeyPrimitive);

        RSAKeyParameters rsaPublicKeyParam = new RSAKeyParameters(
                false,
                rsaPublicKey.getModulus(),
                rsaPublicKey.getPublicExponent()
        );

        return new UserKeysForSupplier(rsaPublicKeyParam, decoder.decode(encodedHmacKey));
    }

    public static TreeMap<String, String> toInterMap(UserKeysForSupplier userKeysForSupplier) throws IOException {
        TreeMap<String, String> interMap = new TreeMap<>();

        RSAKeyParameters asymmetricPublicKey = userKeysForSupplier.getRsaPublicKey();
        RSAPublicKey asn1PublicKey = new RSAPublicKey(asymmetricPublicKey.getModulus(), asymmetricPublicKey.getExponent());

        Base64.Encoder encoder = Base64.getEncoder();

        // get key encoded in DER then base64 for output
        String encodedPublicKey = encoder.encodeToString(asn1PublicKey.getEncoded("DER"));
        String encodedHmacKey = encoder.encodeToString(userKeysForSupplier.getHmacSecretKey());

        interMap.put("rsaPublicKey", encodedPublicKey);
        interMap.put("secretHmacKey", encodedHmacKey);

        // add selfcheck-sum. this isn't for security but to detect corrupt keyfiles as early as possible
        interMap.put("selfcheckSum", getSelfcheckSum(encodedPublicKey, encodedHmacKey));

        return interMap;
    }

    public static TreeMap<String, String> toInterMap(UserKeysForConsumer userKeysForConsumer) throws IOException {
        TreeMap<String, String> interMap = new TreeMap<>();

        RSAKeyParameters asymmetricPrivateKey = userKeysForConsumer.getRsaPrivateKey();
        PrivateKeyInfo privateKeyInfo = PrivateKeyInfoFactory.createPrivateKeyInfo(asymmetricPrivateKey);

        Base64.Encoder encoder = Base64.getEncoder();

        // get key encoded in PKCS8 then base64 for output
        String encodedPrivateKey = encoder.encodeToString(privateKeyInfo.getEncoded());
        String encodedHmacKey = encoder.encodeToString(userKeysForConsumer.getHmacSecretKey());

        interMap.put("rsaPrivateKey", encodedPrivateKey);
        interMap.put("secretHmacKey", encodedHmacKey);

        // add selfcheck-sum. this isn't for security but to detect corrupt keyfiles as early as possible
        interMap.put("selfcheckSum", getSelfcheckSum(encodedPrivateKey, encodedHmacKey));

        return interMap;
    }
}
