/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.notice;

import com.metaeffekt.artifact.analysis.flow.InventoryResult;
import com.metaeffekt.artifact.analysis.utils.StringUtils;
import com.metaeffekt.resource.InventoryResource;
import org.apache.commons.collections4.CollectionUtils;
import org.metaeffekt.common.notice.model.NoticeParameters;
import org.metaeffekt.common.notice.model.ProjectContext;
import org.metaeffekt.core.inventory.processor.model.Artifact;
import org.metaeffekt.core.inventory.processor.model.Inventory;
import org.metaeffekt.notice.NoticeParameterProcessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

import static com.metaeffekt.artifact.analysis.metascan.Constants.KEY_NOTICE_PARAMETER;
import static com.metaeffekt.artifact.analysis.utils.InventoryUtils.joinLicenses;

public class GenerateNoticeFlow {

    private static final Logger LOG = LoggerFactory.getLogger(GenerateNoticeFlow.class);

    public InventoryResult process(GenerateNoticeFlowParam generateNoticeFlowParam) {
        final Inventory inventory = generateNoticeFlowParam.getInventoryResource().getInventory();

        // use derived Notice Parameter (observed); FIXME: perhaps the scan provided a better option; evaluate
        for (Artifact artifact : inventory.getArtifacts()) {
            final String derivedNoticeParameter = artifact.get("Derived Notice Parameter");
            adoptDeductedLicenseAndNoticeParameter(artifact, derivedNoticeParameter);
        }

        // TODO: establish project context
        final ProjectContext projectContext = new ProjectContext();
        projectContext.setLanguageMode(ProjectContext.LanguageMode.en_US);
        projectContext.setSourceMode(ProjectContext.SourceMode.ON_DEMAND);

        // produce notices
        final NoticeParameterProcessor processor = new NoticeParameterProcessor(generateNoticeFlowParam.getNormalizationMetaData());
        processor.updateNoticeFromNoticeParameter(inventory, projectContext);

        return new InventoryResult(InventoryResource.fromInventory(inventory));
    }

    private static void adoptDeductedLicenseAndNoticeParameter(Artifact artifact, String deductedNoticeParameter) {
        // NOTE: if curated license and/or notice parameter information exists, it is never overwritten; it may not 
        //  match the derivedNoticeParameter-defined associated license. But this should then result in an inconsistency
        //  warning that needs to be resolved.

        // first apply notice parameter if no notice parameter is yet set
        if (deductedNoticeParameter != null && StringUtils.isEmpty(artifact.get(KEY_NOTICE_PARAMETER))) {

            // deduct license from current (potentially adjusted) notice parameter; if yet empty
            final NoticeParameters noticeParameters = parseNoticeParameter(artifact, deductedNoticeParameter);

            // we can only adopt a notice parameter that could be available/parsed
            if (noticeParameters != null) {

                // we can only adopt the notice parameter if there is a matching, curated license; otherwise data
                // would become inconsistent
                final HashSet<String> noticeParameterAssociatedLicenses = new HashSet<>(noticeParameters.getAllAssociatedLicense());
                final HashSet<String> currentAssociatedLicenses = new HashSet<>(artifact.getLicenses());
                boolean adopt = (StringUtils.isEmpty(artifact.getLicense()) ? true :
                        CollectionUtils.isEqualCollection(noticeParameterAssociatedLicenses, currentAssociatedLicenses));

                if (adopt) {
                    adoptNoticeParameter(artifact, noticeParameters, deductedNoticeParameter);
                } else {
                    LOG.warn("Cannot adopt deduced notice parameter for [{}]. Inconsistency with curated license.", artifact.deriveQualifier());
                    LOG.warn("Curated:          [{}]", currentAssociatedLicenses.stream().sorted().collect(Collectors.toList()));
                    LOG.warn("Notice Parameter: [{}]", noticeParameterAssociatedLicenses.stream().sorted().collect(Collectors.toList()));
                }
            }
        }
    }

    private static NoticeParameters parseNoticeParameter(Artifact artifact, String derivedNoticeParameterYaml) {
        // update license from notice parameter
        try {
            if (StringUtils.notEmpty(derivedNoticeParameterYaml)) {
                return NoticeParameters.readYaml(derivedNoticeParameterYaml);
            }
        } catch (RuntimeException e) {
            LOG.warn("Cannot parse and process notice parameter for {}.", artifact.deriveQualifier());
        }
        return null;
    }

    private static void adoptNoticeParameter(Artifact artifact, NoticeParameters noticeParameters, String originalNoticeParameterYaml) {
        if (noticeParameters != null) {
            final Set<String> allAssociatedLicense = new HashSet<>(noticeParameters.getAllAssociatedLicense());
            final String associatedLicensesFromNoticeParameter = joinLicenses(allAssociatedLicense);

            // set license and notice parameter pair

            // NOTE: this may overwrite curated license data (consistency checked beforehand), but order and multiple
            //  licenses may be changed.
            artifact.setLicense(associatedLicensesFromNoticeParameter);
            artifact.set(KEY_NOTICE_PARAMETER, originalNoticeParameterYaml);
        }
    }

}
