/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.notice;

import com.metaeffekt.artifact.analysis.metascan.Constants;
import com.metaeffekt.artifact.analysis.utils.InventoryUtils;
import com.metaeffekt.artifact.analysis.utils.StringUtils;
import com.metaeffekt.artifact.terms.model.NormalizationMetaData;
import com.metaeffekt.artifact.terms.model.TermsMetaData;
import lombok.ToString;
import org.metaeffekt.common.notice.model.ComponentDefinition;
import org.metaeffekt.common.notice.model.NoticeParameters;
import org.metaeffekt.core.inventory.processor.model.Artifact;
import org.metaeffekt.core.inventory.processor.model.Inventory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import static org.metaeffekt.core.inventory.InventoryUtils.tokenizeLicense;

/**
 * Builds NoticeParameters based on data in an inventory. Does not require scanner outputs.
 */
public class NoticeParameterBuilderInventory {

    private static final Logger LOG = LoggerFactory.getLogger(NoticeParameterBuilderInventory.class);

    private final NormalizationMetaData normalizationMetaData;

    public NoticeParameterBuilderInventory(NormalizationMetaData normalizationMetaData) {
        this.normalizationMetaData = normalizationMetaData;
    }

    @ToString
    private static class ParameterSet {
        protected String license;
        protected String copyrights;

        protected ParameterSet withLicenseAndCopyrights(String license, String copyrights) {
            this.license = license;
            this.copyrights = copyrights;

            // manage copyrights
            if (copyrights != null && copyrights.startsWith("[")) {
                this.copyrights = null;
            }

            return this;
        }
    }

    public Inventory enrichInventoryWithNoticeParameter(Inventory inventory) throws IOException {
        for (Artifact artifact: inventory.getArtifacts()) {

            if (StringUtils.hasText(artifact.get(Constants.KEY_DERIVED_NOTICE_PARAMETER))) continue;

            // derived (observed)
            ParameterSet derivedObservedParameterSet = new ParameterSet().withLicenseAndCopyrights(
                    artifact.get(Constants.KEY_DERIVED_LICENSES),
                    artifact.get(Constants.KEY_EXTRACTED_COPYRIGHTS_SCANCODE));

            // FIXME: we may want to check the consistency of the parameter sets amongst each other
            final ParameterSet[] parameterSets = {
                derivedObservedParameterSet,
                getDerivedBinaryParameterSet(artifact, "Binary Artifact"),
                getDerivedBinaryParameterSet(artifact, "Source Artifact"),
                getDerivedBinaryParameterSet(artifact, "Descriptor"),
                getDerivedBinaryParameterSet(artifact, "Source Archive")
            };

            for (ParameterSet parameterSet : parameterSets) {
                if (isApplicable(parameterSet)) {
                    final String noticeParameter = buildNoticeParameter(parameterSet);
                    if (noticeParameter != null) {
                        artifact.set(Constants.KEY_DERIVED_NOTICE_PARAMETER, noticeParameter);

                        // stop processing the for loop; once a notice parameter can be produced (priority order)
                        break;
                    }
                }
            }
        }
        return inventory;
    }

    private ParameterSet getDerivedBinaryParameterSet(Artifact artifact, String prefix) {
        return new ParameterSet().withLicenseAndCopyrights(
                artifact.get(prefix + " - Derived License"),
                artifact.get(prefix + " - Extracted Copyrights (ScanCode)"));
    }

    private String buildNoticeParameter(ParameterSet parameterSet){
        final NoticeParameters noticeParameter = new NoticeParameters();
        final ComponentDefinition component = new ComponentDefinition();

        // license
        final String associatedLicense = parameterSet.license;
        final String effectiveLicenses = InventoryUtils.deriveEffectiveLicenses(associatedLicense);

        // map single licese to TermsMetaData
        final TermsMetaData tmd = getEffectiveTermsMetaDataNonNull(effectiveLicenses);

        // copyrights
        if (tmd.getRequiresCopyright() != null && tmd.getRequiresCopyright()) {

            // if copyrights are required but not provided we cannot create a notice parameter and return null
            if (StringUtils.isEmpty(parameterSet.copyrights)) {
                return null;
            }

            String scanCodeCopyrights = parameterSet.copyrights;
            List<String> copyrights = Arrays.asList(scanCodeCopyrights.split("\\|\n"));
            component.setCopyrights(copyrights);
        }

        component.setAssociatedLicenses(tokenizeLicense(parameterSet.license, true, true));
        if (!effectiveLicenses.equals(associatedLicense)){
            component.setEffectiveLicenses(tokenizeLicense(effectiveLicenses, true, true));
        }
        
        noticeParameter.setComponent(component);

        return noticeParameter.toYamlString();
    }

    private boolean isApplicable(ParameterSet artifact) {
        final List<String> licenses = tokenizeLicense(artifact.license, false, false);
        // check licenses
        if (licenses.size() != 1) {
            return false;
        }
        return true;
    }

    public TermsMetaData getEffectiveTermsMetaDataNonNull(String license) {
        final TermsMetaData termsMetaData = normalizationMetaData.findTermsMetaData(license);
        if (termsMetaData == null) {
            return createDefaultTermsMetaData(license);
        } else {
            return termsMetaData;
        }
    }

    public static TermsMetaData createDefaultTermsMetaData(String name){
        final TermsMetaData tmd = new TermsMetaData();
        tmd.setCanonicalName(name);
        tmd.setRequiresNote(true);
        tmd.setRequiresCopyright(false);
        tmd.setTemporaryLMD(true);
        return tmd;
    }
    
}
