/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow.notice;

import com.metaeffekt.artifact.analysis.utils.StringUtils;
import org.metaeffekt.common.notice.model.ComponentDefinition;
import org.metaeffekt.common.notice.model.NoticeParameters;
import org.metaeffekt.core.inventory.processor.model.Artifact;
import org.metaeffekt.core.inventory.processor.model.Inventory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.metaeffekt.artifact.analysis.metascan.Constants.KEY_DERIVED_NOTICE_PARAMETER;

public class NoticeParameterPostProcessing {

    private static final Logger LOG = LoggerFactory.getLogger(NoticeParameterPostProcessing.class);

    public void postProcessNoticeParameter(Inventory inventory) {
        int i = 1;
        for (Artifact artifact : inventory.getArtifacts()) {
            LOG.debug("Postprocessing notice parameter for {} ({}/{})", artifact.deriveQualifier(), i++, inventory.getArtifacts().size());
            try {
                final String derivedNoticeParameter = artifact.get(KEY_DERIVED_NOTICE_PARAMETER);
                if (StringUtils.hasText(derivedNoticeParameter)) {
                    final NoticeParameters noticeParameters = NoticeParameters.readYaml(derivedNoticeParameter);
                    postProcessNoticeParameter(noticeParameters);

                    // update artifact with computed notice parameter yaml
                    artifact.set(KEY_DERIVED_NOTICE_PARAMETER, noticeParameters.toYamlString());
                }
            } catch (Exception e) {
                LOG.warn("Failure processing notice parameter for artifact {}", artifact.getId(), e);
            }
        }
    }

    public NoticeParameters postProcessNoticeParameter(NoticeParameters noticeParameter) {
        if (noticeParameter == null) return null;
        final ComponentDefinition rootComponent = noticeParameter.getComponent();
        if (rootComponent == null) {
            return null;
        }

        // merge components with same license
        final Map<String, ComponentDefinition> associatedLicenseSet = new HashMap<>();
        associatedLicenseSet.put(rootComponent.toStringKey(), rootComponent);
        if (noticeParameter.getSubcomponents() != null) {
            final List<ComponentDefinition> toBeRemoved = new ArrayList<>();
            for (ComponentDefinition subcomponent : noticeParameter.getSubcomponents()) {
                String key = subcomponent.toStringKey();
                if (associatedLicenseSet.containsKey(key)) {
                    ComponentDefinition component = noticeParameter.findComponent(associatedLicenseSet.get(key));

                    // take over copyrights before removing
                    if (component != null && subcomponent.hasCopyrights()) {
                        component.getCopyrights().addAll(subcomponent.getCopyrights());
                    }
                    toBeRemoved.add(subcomponent);
                }
                else {
                    associatedLicenseSet.put(subcomponent.toStringKey(), subcomponent);
                }
            }
            noticeParameter.getSubcomponents().removeAll(toBeRemoved);
        }

        // filter components on duplicates
        filterComponentDefinition(rootComponent);

        // filter subcomponents
        if (noticeParameter.getSubcomponents() != null) {
            for (ComponentDefinition subcomponent : noticeParameter.getSubcomponents()) {
                filterComponentDefinition(subcomponent);
            }
        }

        // filter unassigned copyright information
        if (noticeParameter.getUnassignedInformation() != null) {
            final List<String> copyrights = noticeParameter.getUnassignedInformation().getCopyrights();
            if (copyrights != null) {
                noticeParameter.getUnassignedInformation().setCopyrights(copyrights.stream().distinct().collect(Collectors.toList()));
            }
        }

        return noticeParameter;
    }
    
    private void filterComponentDefinition(ComponentDefinition componentDefinition) {
        componentDefinition.setAssociatedLicenses(componentDefinition.getAssociatedLicenses().stream().distinct().collect(Collectors.toList()));
        componentDefinition.setCopyrights(componentDefinition.getCopyrights().stream().distinct().collect(Collectors.toList()));
    }

}