/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.workbench;

import com.metaeffekt.artifact.analysis.utils.InventorySearch;
import com.metaeffekt.artifact.analysis.utils.InventoryUtils;
import com.metaeffekt.artifact.analysis.utils.StringUtils;
import lombok.Getter;
import lombok.Setter;
import org.metaeffekt.core.inventory.processor.model.Artifact;
import org.metaeffekt.core.inventory.processor.model.Constants;
import org.metaeffekt.core.inventory.processor.model.Inventory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.metaeffekt.artifact.analysis.metascan.Constants.KEY_DERIVED_LICENSES;
import static com.metaeffekt.artifact.analysis.metascan.Constants.KEY_NOTICE_PARAMETER;
import static com.metaeffekt.artifact.analysis.utils.InventoryUtils.*;
import static com.metaeffekt.artifact.analysis.workbench.PackageLicenseTransformer.COMPONENT_SPECIFIED_LICENSE_MAPPED;
import static com.metaeffekt.artifact.analysis.workbench.PackageLicenseTransformer.PACKAGE_SPECIFIED_LICENSE_MAPPED;

public class ProjectInventoryFilter {

    private static final Logger LOG = LoggerFactory.getLogger(ProjectInventoryFilter.class);

    private boolean validate = true;

    @Getter
    @Setter
    private boolean enableUseDerivedLicense = false;

    @Getter
    @Setter
    private boolean enableInherit = true;

    @Getter
    @Setter
    private boolean enableAddComponentNameAndVersion = true;

    @Getter
    @Setter
    private boolean enableApplyCurationData = false;

    public void process(Inventory projectInventory, Inventory referenceInventory) {

        if (enableAddComponentNameAndVersion) {
            // the component name and version are in particular relevant to be able to generate a notice
            deriveComponentNamesAndVersions(projectInventory);
        }

        // perform curation first
        if (enableApplyCurationData) {
            applyCurationData(projectInventory, referenceInventory);
        }

        // inherit information (based on curated data)
        if (enableInherit) {
            InventoryUtils.inheritInformationFromReferenceInventory(projectInventory, new InventorySearch(referenceInventory));
        }

        // fallback to derived licenses if desired
        if (enableUseDerivedLicense) {
            useDerivedLicenses(projectInventory);
        }

        postProcess(projectInventory);
    }

    private void applyCurationData(Inventory inventory, Inventory referenceInventory) {
        if (referenceInventory == null) return;
        final InventorySearch referenceInventorySearch = new InventorySearch(referenceInventory);
        for (final Artifact artifact : inventory.getArtifacts()) {
            artifact.deriveArtifactId();

            // attempt id/checksum request
            Artifact referenceMatch = referenceInventorySearch.findArtifactByIdAndChecksum(artifact);

            if (referenceMatch == null) {
                referenceMatch = referenceInventorySearch.findArtifactById(artifact);
            }

            if (referenceMatch == null) {
                referenceMatch = referenceInventorySearch.findArtifactWildcardMatchingId(artifact);
            }

            if (referenceMatch != null) {

                Artifact referenceClone = new Artifact(referenceMatch);

                // not too clean; but reference is treated read-only anyway
                referenceClone.set("Projects", null);
                referenceClone.set("Path is Asset", null);

                if (Constants.ASTERISK.equals(referenceClone.getVersion())) {
                    referenceClone.setVersion(null);
                }

                // allow overwrites on component, version, license and notice parameter
                if (StringUtils.hasText(referenceClone.getComponent())) {
                    artifact.setComponent(null);
                }
                if (StringUtils.hasText(referenceClone.getVersion())) {
                    artifact.setVersion(null);
                }
                if (StringUtils.hasText(referenceClone.getLicense())) {
                    artifact.setLicense(null);
                }
                if (StringUtils.hasText(referenceClone.get(KEY_NOTICE_PARAMETER))) {
                    artifact.set(KEY_NOTICE_PARAMETER, null);
                }

                // merge all attributes, that are not empty
                artifact.merge(referenceClone);
            }
        }
    }

    private void postProcess(Inventory projectInventory) {
        // transform inventory
        final InventoryTransformer transformer = new InventoryTransformer(getNormalizationMetaData());
        transformer.updateCanonicalNamesInInventory(projectInventory);

        // FIXME: not here!!
        // apply license derivation rules
        shareLicenseInPackageGroup(projectInventory);

        // add license data (always)
        WorkbenchUtils.addLicenseData(projectInventory, false, true);
    }

    private void shareLicenseInPackageGroup(Inventory filteredInventory) {
        final Map<String, String> packageGroupToLicenseMap = new HashMap<>();

        // collect
        for (Artifact artifact : filteredInventory.getArtifacts()) {
            String packageGroup = artifact.get("Package Group");
            if (StringUtils.isEmpty(packageGroup)) packageGroup = artifact.getComponent();
            String license = artifact.getLicense();
            if (StringUtils.hasText(packageGroup) && StringUtils.hasText(license)) {
                packageGroupToLicenseMap.put(packageGroup, license);
            }
        }

        // apply
        for (Artifact artifact : filteredInventory.getArtifacts()) {
            String packageGroup = artifact.get("Package Group");
            if (StringUtils.isEmpty(packageGroup)) packageGroup = artifact.getComponent();
            String license = artifact.getLicense();
            if (StringUtils.hasText(packageGroup) && !StringUtils.hasText(license)) {
                artifact.setLicense(packageGroupToLicenseMap.get(packageGroup));
            }
        }
    }

    /**
     * Implicitly normalizes the licenses
     *
     * @param filteredInventory Inventory to derive licenses from.
     */
    private void useDerivedLicenses(Inventory filteredInventory) {
        for (Artifact artifact : filteredInventory.getArtifacts()) {
            if (StringUtils.isEmpty(artifact.getLicense())) {
                List<String> aggregatedLicenses = new ArrayList<>();

                // FIXME: currently a merge approach; use shared approach; see also spdx exporter

                aggregateLicenses(artifact, KEY_DERIVED_LICENSES, aggregatedLicenses);

                // also consider mapped package licenses if available (no implicit mapping)
                aggregateLicenses(artifact, PACKAGE_SPECIFIED_LICENSE_MAPPED, aggregatedLicenses);
                aggregateLicenses(artifact, COMPONENT_SPECIFIED_LICENSE_MAPPED, aggregatedLicenses);

                aggregateLicenses(artifact, "Binary Artifact - Derived Licenses", aggregatedLicenses);
                aggregateLicenses(artifact, "Source Artifact - Derived Licenses", aggregatedLicenses);
                aggregateLicenses(artifact, "Source Archive - Derived Licenses", aggregatedLicenses);
                aggregateLicenses(artifact, "Descriptor - Derived Licenses", aggregatedLicenses);

                artifact.setLicense(joinLicenses(aggregatedLicenses));
            }
        }
    }

    private void aggregateLicenses(Artifact artifact, String key, List<String> aggregatedLicenses) {
        final String derivedLicense = artifact.get(key);
        if (!StringUtils.isEmpty(derivedLicense)) {
            aggregatedLicenses.addAll(tokenizeLicense(derivedLicense, true, true));
        }
    }

}
