/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.workbench;

import com.metaeffekt.artifact.analysis.flow.*;
import com.metaeffekt.artifact.analysis.model.DefaultPropertyProvider;
import com.metaeffekt.artifact.analysis.model.PropertyProvider;
import com.metaeffekt.artifact.analysis.utils.FileUtils;
import com.metaeffekt.artifact.analysis.utils.InventoryUtils;
import com.metaeffekt.artifact.analysis.utils.PropertyUtils;
import com.metaeffekt.artifact.terms.model.LicenseTextProvider;
import com.metaeffekt.artifact.terms.model.NormalizationMetaData;
import com.metaeffekt.resource.InventoryDirResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;

public class WorkbenchExecution {

    private static final Logger LOG = LoggerFactory.getLogger(WorkbenchExecution.class);

    private final File workbenchBaseDir;

    private final File workbenchControlPropertiesFile;

    private final NormalizationMetaData normalizationMetaData;

    private final LicenseTextProvider licenseTextProvider;

    private File referenceInventoryDir;

    /**
     * The includes pattern to match the inventory files against.<br>
     * By default, this is all <code>.xls</code> files in the <code>inventory</code> directory.
     */
    private String referenceInventoryIncludes = "inventory/*.xls*";

    public WorkbenchExecution(final NormalizationMetaData normalizationMetaData, final LicenseTextProvider licenseTextProvider,
              final File workbenchBaseDir, final File workbenchControlPropertiesFile) {

        InventoryUtils.initialize(normalizationMetaData);

        this.normalizationMetaData = normalizationMetaData;
        this.licenseTextProvider = licenseTextProvider;

        this.workbenchBaseDir = workbenchBaseDir;
        this.workbenchControlPropertiesFile = workbenchControlPropertiesFile;

    }

    public void configureReferenceInventory(File referenceInventoryDir, String referenceInventoryIncludes) {
        this.referenceInventoryDir = referenceInventoryDir;
        this.referenceInventoryIncludes = referenceInventoryIncludes;
    }

    public void executeWorkbench() throws Exception {
        final File scannerInputDir = new File(workbenchBaseDir, "scanner/input");
        final File scannerOutputDir = new File(workbenchBaseDir, "scanner/output");
        final File scannerAnalysisDir = new File(workbenchBaseDir, "scanner/analysis");
        final File scannerTmpBaseDir = new File(workbenchBaseDir, "scanner/analysis/tmp");
        final File scannerStatusBaseDir = new File(workbenchBaseDir, "scanner/status");

        final File documentBaseDir = new File(workbenchBaseDir, "documents");

        // create property provider
        final DefaultPropertyProvider propertyProvider = createPropertyProvider();

        // initialize template for ScanFlowParams
        final ScanFlowParam scanFlowParamTemplate = createScanFlowParam(
                normalizationMetaData, licenseTextProvider, propertyProvider, scannerAnalysisDir);

        final InventoryDirResource referenceInventoryResource =
                InventoryDirResource.attach(referenceInventoryDir, referenceInventoryIncludes);

        final Consumer<File> consumer = new ScanConsumer(
                scannerAnalysisDir, scannerTmpBaseDir, scannerOutputDir,
                referenceInventoryResource, scanFlowParamTemplate);

        // create ObserveFolderFlowParam; pass in property provider
        final ObserveFolderFlowParam param = createObserveFolderParam(
                scannerInputDir, scannerTmpBaseDir, scannerStatusBaseDir, consumer, propertyProvider);

        FileUtils.forceMkdir(param.getObservationDir());
        FileUtils.deleteDirectory(param.getTmpBaseDir());
        FileUtils.forceMkdir(param.getTmpBaseDir());

        final DocumentFlowParam documentFlowParam = createDocumentFlowParam(
                propertyProvider, documentBaseDir, scannerOutputDir);

        // kick off two threads...
        final ExecutorService executor = Executors.newFixedThreadPool(2);

        // thread 1: start observer flow (indirectly also scan flow)
        executor.execute(() -> new ObserveFolderFlow().process(param));

        // thread 2: start document flow independently
        executor.execute(() -> {
            while (true) {
                try {
                    Thread.sleep(2000);
                } catch (InterruptedException e) {
                    // do nothing
                }
                try {
                    new DocumentFlow().process(documentFlowParam);
                } catch (Exception e) {
                    LOG.error(e.getMessage(), e);
                }
            }
        });

        while (!executor.isTerminated()) {
            // use to monitor progress
            Thread.sleep(2000);
        }
    }

    private ObserveFolderFlowParam createObserveFolderParam(File scannerInputDir, File scannerTmpBaseDir,
                    File scannerStatusDir, Consumer<File> consumer, PropertyProvider propertyProvider) {

        final ObserveFolderFlowParam param = new ObserveFolderFlowParam(scannerInputDir, scannerStatusDir, scannerTmpBaseDir, consumer);
        param.configuredBy(propertyProvider);

        return param;
    }

    private ScanFlowParam createScanFlowParam(NormalizationMetaData normalizationMetaData,
          LicenseTextProvider licenseTextProvider, DefaultPropertyProvider propertyProvider, File scannerAnalysisDir) {

        final ScanFlowParam scanFlowParamTemplate = ScanFlowParam.build()
                .using(normalizationMetaData)
                .using(licenseTextProvider)
                .using(scannerAnalysisDir)
                .configuredBy(propertyProvider);
        return scanFlowParamTemplate;
    }

    private DefaultPropertyProvider createPropertyProvider() {
        final DefaultPropertyProvider propertyProvider = new DefaultPropertyProvider();
        final Properties properties = PropertyUtils.loadProperties(workbenchControlPropertiesFile);
        propertyProvider.putAll(properties);
        return propertyProvider;
    }

    public DocumentFlowParam createDocumentFlowParam(PropertyProvider propertyProvider, File documentBaseDir, File scannerOutputDir) {
        // defined by scanner
        final String sourceInventoryPattern = "license-scanner-inventory-*.xls*";

        final File documentOutputDir = new File(documentBaseDir, "output");
        final File documentTempBaseDir = new File(documentBaseDir, "tmp");
        final File documentControlFile = new File(documentBaseDir, "document-control.xlsx");

        final DocumentFlowParam documentFlowParam = new DocumentFlowParam()
                .withDocumentControl(documentControlFile)
                .useScannerOutputDir(scannerOutputDir)
                .useScannerInventoryOutputPattern(sourceInventoryPattern)
                .useTempBaseDir(documentTempBaseDir)
                .publishResultsTo(documentOutputDir)
                .using(normalizationMetaData)
                .using(licenseTextProvider)
                .withReferenceInventory(referenceInventoryDir, referenceInventoryIncludes);

        // allow additions configuration using property provider
        documentFlowParam.configuredBy(propertyProvider);

        return documentFlowParam;
    }

}
