/*
 * Decompiled with CFR 0.152.
 */
package org.metaeffekt.artifact.resolver.maven.index;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.type.TypeFactory;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;
import lombok.NonNull;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.core.WhitespaceAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.WildcardQuery;
import org.metaeffekt.artifact.resolver.download.WebAccess;
import org.metaeffekt.artifact.resolver.generic.utils.MarkerQueryResult;
import org.metaeffekt.artifact.resolver.generic.utils.MarkerUtils;
import org.metaeffekt.artifact.resolver.maven.index.MavenCentralIndexConfig;
import org.metaeffekt.artifact.resolver.maven.index.lucene.MavenLuceneIndex;
import org.metaeffekt.artifact.resolver.maven.index.lucene.MavenLuceneIndexer;
import org.metaeffekt.artifact.resolver.model.DownloadLocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MavenCentralIndex
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(MavenCentralIndex.class);
    private static final int HASH_BUFFER_SIZE = 0x100000;
    private static final String DUMP_HASH_KEY = "DUMP-HASH";
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final WebAccess webAccess;
    private final MavenCentralIndexConfig config;
    private final File dumpFile;
    private final File luceneIndexDir;
    private MavenLuceneIndex mavenLuceneIndex = null;

    public MavenCentralIndex(DownloadLocation downloadLocation, WebAccess webAccess, MavenCentralIndexConfig config) {
        this.webAccess = webAccess;
        this.config = config;
        this.dumpFile = downloadLocation.deriveDownloadFolder("maven-repo-index", "dumps");
        this.luceneIndexDir = downloadLocation.deriveDownloadFolder("maven-repo-index", "index");
    }

    private static String getSha512Hex(InputStream inputStream) throws IOException {
        MessageDigest digest;
        long hashStart = System.currentTimeMillis();
        try {
            digest = MessageDigest.getInstance("SHA-512");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Should never happen: Could not find valid algorithm.", e);
        }
        long contentSize = 0L;
        byte[] buffer = new byte[0x100000];
        int read = inputStream.read(buffer, 0, 0x100000);
        while (read != -1) {
            digest.update(buffer, 0, read);
            contentSize += (long)read;
            read = inputStream.read(buffer, 0, 0x100000);
        }
        String contentHash = Hex.encodeHexString((byte[])digest.digest());
        long hashFinish = System.currentTimeMillis();
        long durationMs = hashFinish - hashStart;
        double durationS = (double)durationMs / 1000.0;
        double contentSizeMiB = contentSize / 0x100000L;
        double megabytesPerSecond = contentSizeMiB / durationS;
        log.debug("Hashed input [{}]MiB in [{}]ms ([{}]MiB/s).", new Object[]{contentSizeMiB, durationMs, String.format("%.2f", megabytesPerSecond)});
        return contentHash;
    }

    private static String getSha512Hex(File toHash) throws IOException {
        try (InputStream inputStream = Files.newInputStream(toHash.toPath(), new OpenOption[0]);){
            String string = MavenCentralIndex.getSha512Hex(inputStream);
            return string;
        }
    }

    private static long fillIndex(MavenLuceneIndexer mavenLuceneIndexer, File jsonDumpGzip) throws IOException {
        AtomicLong total = new AtomicLong(0L);
        log.trace("Clearing index...");
        mavenLuceneIndexer.clear();
        log.trace("Filling index...");
        try (GZIPInputStream inputStream = new GZIPInputStream(Files.newInputStream(jsonDumpGzip.toPath(), new OpenOption[0]), 0x100000);){
            ObjectReader objectReader = new ObjectMapper().readerFor((JavaType)TypeFactory.defaultInstance().constructMapType(HashMap.class, String.class, HashSet.class));
            log.info("Reading maven central index with stream of lines and parallel indexing...");
            try (InputStreamReader streamReader = new InputStreamReader((InputStream)inputStream, StandardCharsets.UTF_8);
                 BufferedReader bufferedReader = new BufferedReader(streamReader, 0x100000);){
                ((Stream)bufferedReader.lines().parallel()).forEach(line -> {
                    try {
                        mavenLuceneIndexer.addEntry((Map)objectReader.readValue(line));
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                    long totalSnapshot = total.incrementAndGet();
                    if (totalSnapshot % 10000000L == 0L) {
                        log.debug("Imported so far: [{}]", (Object)totalSnapshot);
                    }
                });
                log.debug("Committing filled index...");
                mavenLuceneIndexer.commit();
            }
        }
        log.trace("Created [{}] documents.", (Object)total.get());
        return total.get();
    }

    private static void deleteFile(File target) {
        if (!target.delete()) {
            log.warn("Deletion failed unexpectedly: [{}]. Permission issue?", (Object)target.toPath());
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isLuceneIndexCorrelated(String localIndexDumpHash) {
        try (MavenLuceneIndex index = new MavenLuceneIndex(this.luceneIndexDir, (Analyzer)new WhitespaceAnalyzer());){
            boolean bl = !index.lookupContains(DUMP_HASH_KEY, localIndexDumpHash, 64).isEmpty();
            return bl;
        }
        catch (Exception e) {
            log.warn("Irregularity while checking correlation of local dump and local lucene index.");
            return false;
        }
    }

    private synchronized void init() throws IOException {
        log.info("Initializing Maven Central Index...");
        File ndjsonDumpGzip = this.deriveLocalIndexDumpFile();
        String localIndexDumpHash = this.getLocalIndexDumpHash(ndjsonDumpGzip);
        boolean requiresIndexUpdate = this.downloadOrReuseIndexDump(ndjsonDumpGzip, this.luceneIndexDir, localIndexDumpHash);
        if (requiresIndexUpdate |= !this.isLuceneIndexCorrelated(localIndexDumpHash)) {
            this.createIndex(ndjsonDumpGzip, localIndexDumpHash);
        }
        this.mavenLuceneIndex = new MavenLuceneIndex(this.luceneIndexDir, (Analyzer)new WhitespaceAnalyzer());
        this.mavenLuceneIndex.lookupContains("a", "b", 64);
        log.info("Initializing Maven Central Index completed. Operating on [{}] entries.", (Object)this.mavenLuceneIndex.size());
    }

    private void createIndex(File ndjsonDumpGzip, String indexContentHash) throws IOException {
        String debugRef = this.getClass().getSimpleName();
        File luceneDirDummyFile = new File(this.luceneIndexDir, ".DUMMY");
        File markerFile = MarkerUtils.deriveMarkerFileFromDestination(luceneDirDummyFile);
        MarkerQueryResult markerQueryResult = MarkerUtils.queryMarker(markerFile, debugRef);
        if (markerQueryResult.getFoundTarget() == null) {
            long total;
            log.debug("Rebuilding index from json dump...");
            if (markerFile.exists() && !markerFile.delete()) {
                log.warn("Failed to delete marker [{}] while building new index. Might lead to errors.", (Object)markerFile.toPath());
            }
            try (MavenLuceneIndexer mavenLuceneIndexer = new MavenLuceneIndexer(this.luceneIndexDir, (Analyzer)new WhitespaceAnalyzer());){
                total = MavenCentralIndex.fillIndex(mavenLuceneIndexer, ndjsonDumpGzip);
                mavenLuceneIndexer.addEntry(Collections.singletonMap(DUMP_HASH_KEY, Collections.singleton(indexContentHash)));
                try {
                    if (!luceneDirDummyFile.exists() && !luceneDirDummyFile.createNewFile()) {
                        log.warn("Could not create dummy file for marker handling. Issue with lucene destination dir?");
                    }
                    Files.write(luceneDirDummyFile.toPath(), "- dummy content -".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
                    MarkerUtils.markSuccess(luceneDirDummyFile, markerFile, debugRef);
                }
                catch (IOException e) {
                    log.warn("Could not create marker [{}] for built index [{}].", new Object[]{markerFile, debugRef, e});
                }
            }
            catch (Exception e) {
                throw new IOException(e.getMessage(), e);
            }
            log.info("Maven central index constructed with [{}] entries.", (Object)total);
        } else {
            log.debug("Reusing previously initialized index.");
        }
    }

    private String getLocalIndexDumpHash(File localIndexDumpFile) throws IOException {
        log.debug("Hashing input json dump...");
        String indexContentHash = null;
        if (localIndexDumpFile.exists()) {
            try {
                indexContentHash = MavenCentralIndex.getSha512Hex(localIndexDumpFile).trim();
            }
            catch (IOException e) {
                log.error("Error while trying to get hash of existing dump at [{}].", (Object)localIndexDumpFile.toPath());
                throw e;
            }
        }
        return indexContentHash;
    }

    private boolean downloadOrReuseIndexDump(File localIndexDumpFile, File localLuceneIndexDir, String localIndexDumpHash) {
        String remoteHash;
        if (localIndexDumpFile.exists() && !Files.isRegularFile(localIndexDumpFile.toPath(), new LinkOption[0])) {
            log.warn("Local index dump isn't a regular file: [{}]", (Object)localIndexDumpFile.toPath());
            log.info("Reusing possibly outdated local data with sha512 hash [{}].", (Object)localIndexDumpHash);
            return false;
        }
        try (WebAccess.WebSession session = this.webAccess.createSession();){
            String sha512Uri = this.config.getNdjsonDumpGzipUrl() + ".sha512";
            remoteHash = session.downloadToUtf8String(sha512Uri, response -> log.error("Could not get .sha512 file from intended download server [{}]. HTTP status: [{}]", (Object)sha512Uri, (Object)response.getStatusLine().getStatusCode())).map(String::trim).orElse(null);
        }
        catch (IOException e) {
            log.warn("Could not obtain hash of remote index dump.");
            log.info("Reusing possibly outdated local data with sha512 hash [{}].", (Object)localIndexDumpHash);
            return false;
        }
        if (remoteHash == null && localIndexDumpFile.exists()) {
            log.warn("Could not update index. Remote server was not available. Using possibly outdated data!");
            log.info("Reusing possibly outdated local data with sha512 hash [{}].", (Object)localIndexDumpHash);
            return false;
        }
        if (!localIndexDumpFile.exists() || !StringUtils.equals((CharSequence)localIndexDumpHash, (CharSequence)remoteHash)) {
            log.info("Local index dump at [{}] {}. Downloading to [{}].", new Object[]{localIndexDumpFile, localIndexDumpFile.exists() ? "outdated" : "missing", localIndexDumpFile.toPath()});
            File ndjsonDumpGzipMarker = MarkerUtils.deriveMarkerFileFromDestination(localIndexDumpFile);
            if (localIndexDumpFile.exists()) {
                MavenCentralIndex.deleteFile(localIndexDumpFile);
            }
            if (ndjsonDumpGzipMarker.exists()) {
                MavenCentralIndex.deleteFile(ndjsonDumpGzipMarker);
            }
            String debugRef = "maven central index: download to [" + localIndexDumpFile.toPath() + "]";
            File downloaded = MarkerUtils.attemptDownload(this.webAccess, this.config.getNdjsonDumpGzipUrl(), localIndexDumpFile, debugRef);
            if (downloaded == null) {
                log.warn("Could not download source nexus index dump.");
                log.info("Reusing possibly outdated local data with sha512 hash [{}].", (Object)localIndexDumpHash);
                return false;
            }
            try {
                log.debug("Rehashing downloaded index dump to verify download integrity.");
                localIndexDumpHash = MavenCentralIndex.getSha512Hex(localIndexDumpFile);
            }
            catch (IOException e) {
                log.warn("Could not compute hash of downloaded dump at [{}].", (Object)localIndexDumpFile.toPath());
                log.info("Reusing possibly outdated local data with sha512 hash [{}].", (Object)localIndexDumpHash);
                return false;
            }
            if (!StringUtils.equals((CharSequence)localIndexDumpHash, (CharSequence)remoteHash)) {
                log.error("Download to [{}] failed to pass checksum check.", (Object)localIndexDumpFile.toPath());
                MarkerUtils.invalidateMarkerFor(downloaded, debugRef);
                log.info("Reusing possibly outdated local data with sha512 hash [{}].", (Object)localIndexDumpHash);
                return false;
            }
            return true;
        }
        log.debug("Using existing dump from [{}].", (Object)localIndexDumpFile.toPath());
        if (localIndexDumpHash == null) {
            log.warn("Hash of index content null; must be previous failure while downloading index dump.");
            return true;
        }
        return !localLuceneIndexDir.exists();
    }

    @NonNull
    private File deriveLocalIndexDumpFile() {
        if (this.config.getNdjsonDumpGzipPath() != null) {
            return new File(this.config.getNdjsonDumpGzipPath());
        }
        if (this.config.getNdjsonDumpGzipUrl() != null) {
            return new File(this.dumpFile, "downloaded-index-dump.ndjson.gz");
        }
        log.debug("Aborting initialization: insufficient configuration.");
        throw new IllegalArgumentException("Could not derive ndjsonDumpGzip, as paths were missing in config.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void ensureInitialized() throws IOException {
        AtomicBoolean atomicBoolean = this.initialized;
        synchronized (atomicBoolean) {
            if (!this.initialized.get()) {
                log.debug("Attempting index initialization...");
                this.init();
                this.initialized.set(true);
            }
            if (this.mavenLuceneIndex == null) {
                throw new IOException("Index is not properly initialized: lucene index is not available.");
            }
        }
    }

    @NonNull
    public List<Document> queryByGAV(String groupId, @NonNull String artifactId, String version, String classifier, String extensionOrPackaging, int n) throws IOException {
        if (artifactId == null) {
            throw new NullPointerException("artifactId is marked non-null but is null");
        }
        this.ensureInitialized();
        BooleanQuery.Builder builder = new BooleanQuery.Builder();
        if (groupId != null) {
            builder.add((Query)new TermQuery(new Term("g", groupId)), BooleanClause.Occur.MUST);
        }
        if (artifactId != null) {
            builder.add((Query)new TermQuery(new Term("a", artifactId)), BooleanClause.Occur.MUST);
        }
        if (version != null) {
            builder.add((Query)new TermQuery(new Term("v", version)), BooleanClause.Occur.MUST);
        }
        if (classifier != null) {
            builder.add((Query)new TermQuery(new Term("c", classifier)), BooleanClause.Occur.MUST);
        }
        if (extensionOrPackaging != null) {
            builder.add((Query)new TermQuery(new Term("p", extensionOrPackaging)), BooleanClause.Occur.MUST);
        }
        BooleanQuery bq = builder.build();
        return this.mavenLuceneIndex.runQuery((Query)bq, n);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void close() throws Exception {
        AtomicBoolean atomicBoolean = this.initialized;
        synchronized (atomicBoolean) {
            if (this.mavenLuceneIndex != null) {
                this.mavenLuceneIndex.close();
            }
        }
    }

    public List<Document> queryByFilename(@NonNull String filenameQueryString, int n) throws IOException {
        if (filenameQueryString == null) {
            throw new NullPointerException("filenameQueryString is marked non-null but is null");
        }
        this.ensureInitialized();
        BooleanQuery.Builder builder = new BooleanQuery.Builder();
        builder.add((Query)new WildcardQuery(new Term("df", filenameQueryString)), BooleanClause.Occur.SHOULD);
        builder.add((Query)new WildcardQuery(new Term("ef", filenameQueryString)), BooleanClause.Occur.SHOULD);
        return this.mavenLuceneIndex.runQuery((Query)builder.build(), n);
    }

    public static String sanitizeForWildcardQuery(String literalString) {
        return literalString.replace("\\", "\\\\").replace("*", "\\*").replace("?", "\\?");
    }
}

