/*
 * Decompiled with CFR 0.152.
 */
package org.metaeffekt.artifact.resolver.alpine;

import io.fabric8.kubernetes.api.model.EnvVar;
import io.fabric8.kubernetes.client.Config;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import lombok.NonNull;
import org.apache.commons.io.file.PathUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.metaeffekt.artifact.resolver.ArtifactResolverConfig;
import org.metaeffekt.artifact.resolver.alpine.AlpineEnvironmentParameters;
import org.metaeffekt.artifact.resolver.alpine.AlpinePackageReference;
import org.metaeffekt.artifact.resolver.alpine.AlpinePackageResolverConfig;
import org.metaeffekt.artifact.resolver.alpine.AlpineUtils;
import org.metaeffekt.artifact.resolver.alpine.bodge.CurlDownloadReference;
import org.metaeffekt.artifact.resolver.alpine.bodge.StringPieceAssembler;
import org.metaeffekt.artifact.resolver.download.ProxyConfig;
import org.metaeffekt.artifact.resolver.manager.execenv.DownloadEnvironment;
import org.metaeffekt.artifact.resolver.manager.execenv.EnvironmentParameters;
import org.metaeffekt.artifact.resolver.manager.execenv.exception.EnvironmentInitializationFailure;
import org.metaeffekt.artifact.resolver.manager.execenv.exception.EnvironmentOperationFailure;
import org.metaeffekt.core.container.control.ExecutorUtils;
import org.metaeffekt.core.container.control.exception.CommandExecutionFailed;
import org.metaeffekt.core.container.control.kubernetesapi.KubernetesCommandExecutor;
import org.metaeffekt.core.container.control.kubernetesapi.KubernetesContainerCommandProcess;
import org.metaeffekt.core.container.control.kubernetesapi.NamespaceHandle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AlpineEnvironment
implements DownloadEnvironment<AlpinePackageReference> {
    private static final Logger LOG = LoggerFactory.getLogger(AlpineEnvironment.class);
    private final AlpineEnvironmentParameters params;
    private final KubernetesCommandExecutor executor;
    private final Map<String, String> pkgNameVerToAportsPath;
    private boolean closed = false;

    public AlpineEnvironment(Config kubeconfig, @NonNull NamespaceHandle namespaceHandle, @NonNull EnvironmentParameters<AlpineEnvironment> params) throws Exception {
        this(kubeconfig, namespaceHandle, (AlpineEnvironmentParameters)params);
        if (namespaceHandle == null) {
            throw new NullPointerException("namespaceHandle is marked non-null but is null");
        }
        if (params == null) {
            throw new NullPointerException("params is marked non-null but is null");
        }
    }

    public AlpineEnvironment(Config kubeconfig, @NonNull NamespaceHandle namespaceHandle, @NonNull AlpineEnvironmentParameters params) throws Exception {
        if (namespaceHandle == null) {
            throw new NullPointerException("namespaceHandle is marked non-null but is null");
        }
        if (params == null) {
            throw new NullPointerException("params is marked non-null but is null");
        }
        this.params = params;
        ArrayList<EnvVar> envVars = new ArrayList<EnvVar>();
        ProxyConfig proxyConfig = params.getProxyConfig();
        if (proxyConfig != null) {
            String proxyString = proxyConfig.getProxyString();
            envVars.add(new EnvVar("HTTP_PROXY", proxyString, null));
            envVars.add(new EnvVar("HTTPS_PROXY", proxyString, null));
            envVars.add(new EnvVar("NO_PROXY", proxyConfig.getNonProxyHosts(), null));
        }
        this.executor = new KubernetesCommandExecutor(kubeconfig, namespaceHandle.getName(), params.getImageIdentifier(), envVars);
        AlpineEnvironment.runInitCommand(this.executor, "apk", "add", "git", "coreutils", "curl", "abuild", "tar");
        AlpineEnvironment.runInitCommand(this.executor, "git", "clone", "--depth", "1", "--branch", params.getAlpineSourcesTag(), "https://git.alpinelinux.org/aports", "/aports");
        this.pkgNameVerToAportsPath = Collections.unmodifiableMap(AlpineEnvironment.buildAlpinePackageIndexWithAbuild(this.executor));
    }

    private static void runInitCommand(@NonNull KubernetesCommandExecutor executor, String ... command) throws EnvironmentInitializationFailure {
        if (executor == null) {
            throw new NullPointerException("executor is marked non-null but is null");
        }
        if (command == null) {
            throw new NullPointerException("command is marked non-null but is null");
        }
        try {
            int exitValue;
            LOG.trace("Running command [{}].", (Object)Arrays.toString(command));
            try (KubernetesContainerCommandProcess process = executor.executeCommand(command);){
                process.waitFor(10L, TimeUnit.MINUTES);
                exitValue = process.exitValue();
                if (exitValue != 0) {
                    LOG.error("stdout: [{}]", (Object)new String(process.getAllStdOut(), StandardCharsets.UTF_8));
                    LOG.error("stderr: [{}]", (Object)new String(process.getAllStdErr(), StandardCharsets.UTF_8));
                }
            }
            if (exitValue != 0) {
                LOG.error("Failed to execute init command [{}]: exit value [{}].", (Object)Arrays.toString(command), (Object)exitValue);
                throw new EnvironmentInitializationFailure("Failed to execute init command: exit value " + exitValue);
            }
        }
        catch (Exception e) {
            throw new EnvironmentInitializationFailure("Failed to run init command", e);
        }
    }

    private static Map<String, String> buildAlpinePackageIndexWithAbuild(KubernetesCommandExecutor executor) throws EnvironmentInitializationFailure {
        HashMap<String, String> nameVerToPath = new HashMap<String, String>();
        try {
            Path tempDir = Files.createTempDirectory("aecc-", new FileAttribute[0]);
            Path packagesListingPath = tempDir.resolve("subpackagesListing.txt");
            AlpineEnvironment.runInitCommand(executor, "sh", "-c", "find /aports -type f -name 'APKBUILD' -print -o -type d -name \"testing\" -prune > /aports/apkbuildList.txt");
            LOG.debug("Starting to generate subpackage names (this may take a while)...");
            AlpineEnvironment.runInitCommand(executor, "sh", "-c", "cat /aports/apkbuildList.txt | sed \"s/APKBUILD\\$//g\" | while read i; do cd \"$i\" ; printf \"%s \" \"$PWD\"; abuild -F listpkg | tr \"\\n\" \" \"; printf \"\\n\"; done > /aports/subpackagesListing.txt");
            executor.downloadFile("/aports/subpackagesListing.txt", packagesListingPath);
            if (!packagesListingPath.toFile().exists()) {
                throw new EnvironmentInitializationFailure("Panic: subpackagesListing.txt could not be transferred.");
            }
            if (!Files.isRegularFile(packagesListingPath, new LinkOption[0])) {
                throw new EnvironmentInitializationFailure("Panic: subpackagesListing.txt was not a regular file.");
            }
            if (Files.size(packagesListingPath) < 1L) {
                throw new EnvironmentInitializationFailure("Panic: Transferred subpackagesListing.txt appears to be empty");
            }
            if (Files.readAllLines(packagesListingPath).size() < 512) {
                LOG.error("subpackages listing file output: [{}]", Files.readAllLines(packagesListingPath));
                throw new EnvironmentInitializationFailure("Panic: Less than 512 packages found. That can't be right.");
            }
            try (Stream<String> lineStream = Files.lines(packagesListingPath, StandardCharsets.UTF_8);){
                Map pathToNameVerMap = lineStream.collect(HashMap::new, AlpineUtils::parseAbuildIndexLine, HashMap::putAll);
                for (Map.Entry pathToNameVer : pathToNameVerMap.entrySet()) {
                    for (String nameVer : (List)pathToNameVer.getValue()) {
                        String previousPath = (String)nameVerToPath.putIfAbsent(nameVer, (String)pathToNameVer.getKey());
                        if (previousPath == null) continue;
                        LOG.warn("Ambiguous name-ver [{}] occurs in: [{}], also in [{}]. Removing from resolver.", new Object[]{nameVer, previousPath, pathToNameVer.getKey()});
                    }
                }
            }
            PathUtils.deleteDirectory((Path)tempDir);
        }
        catch (IOException e) {
            throw new EnvironmentInitializationFailure("Error while preparing APKBUILD index", e);
        }
        return nameVerToPath;
    }

    private void ensureOpen() throws IllegalStateException {
        if (this.closed) {
            throw new IllegalStateException("Environment has already been destroyed.");
        }
    }

    private static String squote(String input) {
        return "'" + input.replaceAll("'", "\"'\"") + "'";
    }

    private static String[] shellify(String absolutePath, @NonNull String command) {
        if (command == null) {
            throw new NullPointerException("command is marked non-null but is null");
        }
        return new String[]{"sh", "-c", "cd " + AlpineEnvironment.squote(absolutePath) + " && " + command};
    }

    private synchronized void abuildFetch(@NonNull String absolutePath) throws CommandExecutionFailed {
        if (absolutePath == null) {
            throw new NullPointerException("absolutePath is marked non-null but is null");
        }
        String[] fetchCommand = AlpineEnvironment.shellify(absolutePath, "abuild -F fetch");
        ExecutorUtils.demandSuccess((KubernetesCommandExecutor)this.executor, (String[])fetchCommand, (long)10L, (TimeUnit)TimeUnit.MINUTES);
    }

    private synchronized void abuildVerify(@NonNull String absolutePath) throws CommandExecutionFailed {
        if (absolutePath == null) {
            throw new NullPointerException("absolutePath is marked non-null but is null");
        }
        String[] verifyCommand = AlpineEnvironment.shellify(absolutePath, "abuild -F verify");
        ExecutorUtils.demandSuccess((KubernetesCommandExecutor)this.executor, (String[])verifyCommand, (long)5L, (TimeUnit)TimeUnit.MINUTES);
    }

    private synchronized void abuildSrcpkg(@NonNull String absolutePath) throws CommandExecutionFailed {
        if (absolutePath == null) {
            throw new NullPointerException("absolutePath is marked non-null but is null");
        }
        String[] srcpkgCommand = AlpineEnvironment.shellify(absolutePath, "abuild -F -P " + AlpineEnvironment.squote(absolutePath + "/ae-out") + " srcpkg");
        ExecutorUtils.demandSuccess((KubernetesCommandExecutor)this.executor, (String[])srcpkgCommand, (long)2L, (TimeUnit)TimeUnit.MINUTES);
    }

    private synchronized void curlDownloadSource(@NonNull String absolutePath, @NonNull String outputFilename, @NonNull String url) throws CommandExecutionFailed {
        if (absolutePath == null) {
            throw new NullPointerException("absolutePath is marked non-null but is null");
        }
        if (outputFilename == null) {
            throw new NullPointerException("outputFilename is marked non-null but is null");
        }
        if (url == null) {
            throw new NullPointerException("url is marked non-null but is null");
        }
        Object[] curlDownloadCommand = AlpineEnvironment.shellify(absolutePath, "curl -L -o " + AlpineEnvironment.squote(outputFilename) + " " + AlpineEnvironment.squote(url));
        LOG.trace("Running Curl download [{}].", (Object)Arrays.toString(curlDownloadCommand));
        ExecutorUtils.demandSuccess((KubernetesCommandExecutor)this.executor, (String[])curlDownloadCommand, (long)1L, (TimeUnit)TimeUnit.MINUTES);
        LOG.trace("Custom curl download SUCCESS: [{}].", (Object)Arrays.toString(curlDownloadCommand));
    }

    private synchronized String getSrcpkgPath(@NonNull String absolutePath) throws EnvironmentOperationFailure, CommandExecutionFailed {
        String stdOut;
        if (absolutePath == null) {
            throw new NullPointerException("absolutePath is marked non-null but is null");
        }
        Object[] getPkgPathCommand = AlpineEnvironment.shellify("/", "for file in " + AlpineEnvironment.squote(absolutePath) + "/ae-out/src/* ; do printf \"%s\\0\" \"$file\" ; done");
        try (KubernetesContainerCommandProcess process = this.executor.executeCommand((String[])getPkgPathCommand);){
            process.waitFor(1L, TimeUnit.MINUTES);
            if (process.exitValue() != 0) {
                LOG.error("Command execution of command [{}] failed with exit value [{}].", (Object)Arrays.toString(getPkgPathCommand), (Object)process.exitValue());
                throw new CommandExecutionFailed("Command execution returned with non-zero exit code");
            }
            stdOut = new String(process.getAllStdOut(), StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            throw new CommandExecutionFailed("Could not get srcpkg path because command failed", e);
        }
        if (stdOut.indexOf("\u0000") == stdOut.lastIndexOf("\u0000") && StringUtils.isNotBlank((CharSequence)stdOut)) {
            return stdOut.substring(0, stdOut.length() - 1);
        }
        LOG.error("Failing resolution: multiple files in srcpkg outdir of [{}], expected one.", (Object)absolutePath);
        throw new EnvironmentOperationFailure("Multiple files found in srcpkg outdir of [{}], expected one.");
    }

    private synchronized String doDownloadAndSrcpkg(@NonNull String aportsPath) throws CommandExecutionFailed, EnvironmentOperationFailure {
        if (aportsPath == null) {
            throw new NullPointerException("aportsPath is marked non-null but is null");
        }
        this.ensureOpen();
        LOG.debug("Doing downloads for package at [{}]", (Object)aportsPath);
        this.abuildFetch(aportsPath);
        this.abuildVerify(aportsPath);
        this.abuildSrcpkg(aportsPath);
        return this.getSrcpkgPath(aportsPath);
    }

    private synchronized void tryCustomDownload(@NonNull String aportsPath, @NonNull AlpinePackageReference packageRef, @NonNull AlpinePackageResolverConfig resolverConfig) throws CommandExecutionFailed {
        if (aportsPath == null) {
            throw new NullPointerException("aportsPath is marked non-null but is null");
        }
        if (packageRef == null) {
            throw new NullPointerException("packageRef is marked non-null but is null");
        }
        if (resolverConfig == null) {
            throw new NullPointerException("resolverConfig is marked non-null but is null");
        }
        List<CurlDownloadReference> downloads = resolverConfig.getExplicitSourceDownloads().get(packageRef.getPkgname());
        if (downloads.isEmpty()) {
            LOG.trace("Can't do anything to fix up missing sources for package ref [{}].", (Object)packageRef);
            return;
        }
        for (CurlDownloadReference download : downloads) {
            String outputFilename = StringPieceAssembler.createString(download.getOutputFilenamePieces(), packageRef);
            String downloadUrl = StringPieceAssembler.createString(download.getUrlPieces(), packageRef);
            if (outputFilename.contains("..")) {
                LOG.warn("Potentially problematic filename [{}] with '..' generated for package ref [{}].", (Object)outputFilename, (Object)packageRef);
            }
            LOG.info("Downloading manually specified sources for ref [{}] from [{}] to [{}]", new Object[]{packageRef, downloadUrl, outputFilename});
            this.curlDownloadSource(aportsPath, outputFilename, downloadUrl);
        }
    }

    @Override
    public synchronized void getSourceArchive(@NonNull AlpinePackageReference reference, @NonNull Path outputPath, @NonNull ArtifactResolverConfig resolverConfig) throws EnvironmentOperationFailure {
        String aportsPath;
        if (reference == null) {
            throw new NullPointerException("reference is marked non-null but is null");
        }
        if (outputPath == null) {
            throw new NullPointerException("outputPath is marked non-null but is null");
        }
        if (resolverConfig == null) {
            throw new NullPointerException("resolverConfig is marked non-null but is null");
        }
        this.ensureOpen();
        if (!this.params.getAlpineSourcesTag().equals("v" + reference.getAlpineVersion())) {
            LOG.warn("Using alpine environment with aports tag [{}] with a reference that requested [{}].", (Object)this.params.getAlpineSourcesTag(), (Object)reference.getAlpineVersion());
        }
        if ((aportsPath = this.pkgNameVerToAportsPath.get(reference.getPkgname() + "-" + reference.getPkgver())) == null) {
            LOG.error("No alpine package found for ref [{}]", (Object)reference);
            LOG.trace("Map was: [{}]", this.pkgNameVerToAportsPath);
            throw new EnvironmentOperationFailure("Could not find aports path for given reference.");
        }
        String pathInContainer = null;
        try {
            pathInContainer = this.doDownloadAndSrcpkg(aportsPath);
        }
        catch (Exception e) {
            LOG.info("First download attempt for [{}] failed with exception: [{}]", (Object)reference, (Object)ExceptionUtils.getStackTrace((Throwable)e));
        }
        try {
            if (pathInContainer == null) {
                LOG.info("Trying custom download for [{}].", (Object)reference);
                this.tryCustomDownload(aportsPath, reference, resolverConfig.getAlpinePackageResolverConfig());
                pathInContainer = this.doDownloadAndSrcpkg(aportsPath);
            }
        }
        catch (Exception e) {
            LOG.warn("Second download attempt for [{}] failed with exception: [{}]", (Object)reference, (Object)ExceptionUtils.getStackTrace((Throwable)e));
        }
        if (pathInContainer == null) {
            throw new EnvironmentOperationFailure("Could not create source archive.");
        }
        try {
            LOG.debug("Starting transfer from container's [{}] to host's [{}].", (Object)pathInContainer, (Object)outputPath);
            this.executor.downloadFile(pathInContainer, outputPath);
        }
        catch (Exception e) {
            LOG.error("Failed to transfer the source archive for [{}].", (Object)reference);
            throw new EnvironmentOperationFailure("Environment operation failed", e);
        }
        try {
            if (!Files.isRegularFile(outputPath, new LinkOption[0])) {
                LOG.error("Transferred file did not create a regular file.");
                throw new EnvironmentOperationFailure("Mustn't happen: Transferred output was not a regular file.");
            }
            long size = Files.size(outputPath);
            if (size < 1L) {
                LOG.error("Transferred file [{}] appears to be empty (size [{}]).", (Object)outputPath, (Object)size);
                throw new EnvironmentOperationFailure("Mustn't happen: transferred file appears to be empty.");
            }
        }
        catch (IOException e) {
            throw new EnvironmentOperationFailure("Operation failed due to unexpected Exception.", e);
        }
    }

    @Override
    public synchronized void close() throws Exception {
        if (!this.closed) {
            this.closed = true;
            this.executor.close();
        }
    }
}

