/*
 * Decompiled with CFR 0.152.
 */
package org.metaeffekt.artifact.resolver.generic.utils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.function.Supplier;
import lombok.NonNull;
import org.apache.commons.io.IOUtils;
import org.metaeffekt.artifact.resolver.download.WebAccess;
import org.metaeffekt.artifact.resolver.generic.utils.GenericUtils;
import org.metaeffekt.artifact.resolver.generic.utils.MarkerQueryResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MarkerUtils {
    private static final Logger log = LoggerFactory.getLogger(MarkerUtils.class);

    public static void touchMarker(@NonNull File markerFile, String reference) {
        if (markerFile == null) {
            throw new NullPointerException("markerFile is marked non-null but is null");
        }
        if (markerFile.exists() && !markerFile.delete()) {
            log.warn("Could not delete existing marker [{}]. Wrong permissing? Ignoring.", (Object)markerFile);
        }
        if (!markerFile.getParentFile().exists() && !markerFile.getParentFile().mkdirs()) {
            log.warn("Could not find nor create marker directory [{}] (via mkdirs). Ignoring.", (Object)markerFile.getParentFile().getAbsolutePath());
        }
        try {
            if (!markerFile.createNewFile()) {
                log.warn("Could not create new marker [{}] while resolving [{}]. Ignoring.", (Object)markerFile, (Object)reference);
            }
        }
        catch (IOException e) {
            log.warn("IOException while creating marker [{}] while resolving [{}]. Ignoring.", (Object)markerFile, (Object)reference);
        }
    }

    public static File deriveMarkerFileFromDestination(@NonNull File nonMarkerFile) {
        if (nonMarkerFile == null) {
            throw new NullPointerException("nonMarkerFile is marked non-null but is null");
        }
        File markerDir = new File(nonMarkerFile.getParentFile(), "MARKER");
        if (markerDir.exists() && !markerDir.isDirectory()) {
            log.error("Reserved marker directory at [{}] exists but isn't a directory. This will likely fail!", (Object)markerDir);
        }
        return new File(markerDir, "[" + nonMarkerFile.getName() + "]");
    }

    public static void invalidateMarkerFor(@NonNull File targetFile, String reference) {
        if (targetFile == null) {
            throw new NullPointerException("targetFile is marked non-null but is null");
        }
        File derivedMarker = MarkerUtils.deriveMarkerFileFromDestination(targetFile);
        MarkerUtils.markFailure(derivedMarker, reference);
    }

    public static void markFailure(@NonNull File markerFile, String reference) {
        if (markerFile == null) {
            throw new NullPointerException("markerFile is marked non-null but is null");
        }
        try {
            if (!markerFile.getParentFile().exists() && !markerFile.getParentFile().mkdirs()) {
                log.error("Could not create parent dir for marker [{}] while resolving for ref [{}].", (Object)markerFile, (Object)reference);
                throw new IOException("Could not create marker's parent dir.");
            }
            if (markerFile.exists() && !Files.isRegularFile(markerFile.toPath(), new LinkOption[0])) {
                log.error("Marker file at [{}] was not a regular file while resolving for ref [{}].", (Object)markerFile, (Object)reference);
                throw new IOException("Could not write invalid marker to non-file.");
            }
            log.trace("Writing invalid marker to [{}] while resolving for ref [{}].", (Object)markerFile, (Object)reference);
            Files.write(markerFile.toPath(), new byte[0], StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);
        }
        catch (IOException e) {
            log.error("Failed to truncate marker file at [{}] while resolving [{}].", (Object)markerFile, (Object)reference);
            throw new RuntimeException(e);
        }
    }

    public static void markSuccess(@NonNull File dest, @NonNull File markerFile, String reference) {
        if (dest == null) {
            throw new NullPointerException("dest is marked non-null but is null");
        }
        if (markerFile == null) {
            throw new NullPointerException("markerFile is marked non-null but is null");
        }
        try {
            MarkerUtils.touchMarker(markerFile, reference);
            Files.write(markerFile.toPath(), dest.toPath().toAbsolutePath().toString().getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            log.warn("Could not write success to marker [{}] for ref [{}]. Future resolve may fail unexpectedly.", (Object)markerFile, (Object)reference);
        }
    }

    public static boolean sanityCheckOutputExists(@NonNull File fileToCheck, File correspondingMarker, String reference) {
        if (fileToCheck == null) {
            throw new NullPointerException("fileToCheck is marked non-null but is null");
        }
        try {
            if (!fileToCheck.exists() || !Files.isRegularFile(fileToCheck.toPath(), new LinkOption[0]) || Files.size(fileToCheck.toPath()) <= 0L) {
                log.warn("Previous output file [{}] of marker [{}] for ref [{}] failed sanity checks.", new Object[]{fileToCheck, correspondingMarker, reference});
                return false;
            }
            return true;
        }
        catch (IOException e) {
            log.trace("IOException while sanity checking existence of file [{}].", (Object)fileToCheck);
            return false;
        }
    }

    @NonNull
    public static MarkerQueryResult queryMarker(@NonNull File markerFile, String reference) {
        long markerSize;
        if (markerFile == null) {
            throw new NullPointerException("markerFile is marked non-null but is null");
        }
        if (!markerFile.exists()) {
            return new MarkerQueryResult(null, false);
        }
        try {
            markerSize = Files.size(markerFile.toPath());
        }
        catch (IOException e) {
            log.error("Error getting size of marker file [{}].", (Object)markerFile);
            return new MarkerQueryResult(null, true);
        }
        if (markerSize > 0L) {
            File previousOutput;
            try (InputStream inputStream = Files.newInputStream(markerFile.toPath(), new OpenOption[0]);){
                previousOutput = new File(IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8));
            }
            catch (IOException e) {
                log.error("IOException reading output path from marker file [{}]. Try removing broken markers.", (Object)markerFile);
                return new MarkerQueryResult(null, true);
            }
            if (!MarkerUtils.sanityCheckOutputExists(previousOutput, markerFile, reference)) {
                log.debug("Returning invalidated marker due to failed sanity checks.");
                MarkerUtils.invalidateMarkerFor(previousOutput, reference);
                return new MarkerQueryResult(null, MarkerUtils.hasRecentDownloadAttempt(markerFile));
            }
            return new MarkerQueryResult(previousOutput, true);
        }
        log.trace("Marker file for [{}] exists.", (Object)reference);
        return new MarkerQueryResult(null, GenericUtils.isModifiedInLast24Hours(markerFile));
    }

    public static boolean hasRecentDownloadAttempt(@NonNull File markerFile) {
        if (markerFile == null) {
            throw new NullPointerException("markerFile is marked non-null but is null");
        }
        return GenericUtils.isModifiedInLast24Hours(markerFile);
    }

    public static File attemptDownload(@NonNull File markerFile, @NonNull String reference, @NonNull Supplier<File> downloadRunner) {
        if (markerFile == null) {
            throw new NullPointerException("markerFile is marked non-null but is null");
        }
        if (reference == null) {
            throw new NullPointerException("reference is marked non-null but is null");
        }
        if (downloadRunner == null) {
            throw new NullPointerException("downloadRunner is marked non-null but is null");
        }
        MarkerQueryResult queryResult = MarkerUtils.queryMarker(markerFile, reference);
        if (queryResult.getFoundTarget() != null) {
            log.debug("Skipping successful download for ref [{}] with marker [{}]; marker gave [{}].", new Object[]{reference, markerFile.toPath(), queryResult.getFoundTarget()});
            return queryResult.getFoundTarget();
        }
        if (queryResult.isAttemptedRecently()) {
            log.info("Skipping failed download for ref [{}] for marker [{}] failed recently.", (Object)reference, (Object)markerFile.toPath());
            return null;
        }
        File downloaded = downloadRunner.get();
        if (downloaded != null) {
            MarkerUtils.markSuccess(downloaded, markerFile, reference);
            return downloaded;
        }
        log.error("Invalidating marker [{}] for ref [{}].", (Object)markerFile.toPath(), (Object)reference);
        MarkerUtils.markFailure(markerFile, reference);
        log.debug("Failed download for ref [{}] and marker [{}].", (Object)reference, (Object)markerFile);
        return null;
    }

    public static File attemptDownload(WebAccess webAccess, String url, File destination, String debugRef) {
        return MarkerUtils.attemptDownload(MarkerUtils.deriveMarkerFileFromDestination(destination), debugRef, () -> GenericUtils.downloadFile(webAccess, url, destination, debugRef));
    }
}

