// Copyright (c) Microsoft Corporation.
// All rights reserved.
//
// This code is licensed under the MIT License.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.microsoft.aad.adal;

import android.accounts.AuthenticatorException;
import android.accounts.OperationCanceledException;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Looper;
import android.os.NetworkOnMainThreadException;
import android.support.v4.content.LocalBroadcastManager;
import android.util.Log;
import android.util.SparseArray;

import com.microsoft.aad.adal.AuthenticationRequest.UserIdentifierType;

import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.atomic.AtomicReference;

/**
 * ADAL context to get access token, refresh token, and lookup from cache.
 */
public class AuthenticationContext {

    private static final int EXCLUDE_INDEX = 8;

    private static final String TAG = "AuthenticationContext";

    private Context mContext;

    private String mAuthority;

    private boolean mValidateAuthority;

    private boolean mIsAuthorityValidated;

    private ITokenCacheStore mTokenCacheStore;

    private BrokerProxy mBrokerProxy = null;

    private boolean mExtendedLifetimeEnabled = false;

    /**
     * Delegate map is needed to handle activity recreate without asking
     * developer to handle context instance for config changes.
     */
    private final SparseArray<AuthenticationRequestState> mDelegateMap = new SparseArray<>();

    /**
     * CorrelationId set by user or generated by ADAL.
     */
    private UUID mRequestCorrelationId = null;

    /**
     * Constructs context to use with known authority to get the token. It uses
     * default cache that stores encrypted tokens.
     * 
     * @param appContext It needs to have handle to the {@link Context} to use
     *            the SharedPreferences as a Default cache storage. It does not
     *            need to be activity.
     * @param authority Authority url to send code and token requests
     * @param validateAuthority validate authority before sending token request
     */
    public AuthenticationContext(Context appContext, String authority, boolean validateAuthority) {
        // Fixes are required for SDK 16-18
        // The fixes need to be applied before any use of Java Cryptography
        // Architecture primitives. Default cache uses encryption
        PRNGFixes.apply();
        initialize(appContext, authority, new DefaultTokenCacheStore(appContext), validateAuthority, true);
    }

    /**
     * Constructs context to use with known authority to get the token. It uses
     * provided cache.
     * 
     * @param appContext {@link Context}
     * @param authority Authority Url
     * @param validateAuthority true/false for validation
     * @param tokenCacheStore Set to null if you don't want cache.
     */
    public AuthenticationContext(Context appContext, String authority, boolean validateAuthority,
            ITokenCacheStore tokenCacheStore) {
        initialize(appContext, authority, tokenCacheStore, validateAuthority, false);
    }

    /**
     * It will verify the authority and use the given cache. If cache is null,
     * it will not use cache.
     * 
     * @param appContext {@link Context}
     * @param authority Authority Url
     * @param tokenCacheStore Cache {@link ITokenCacheStore} used to store
     *            tokens. Set to null if you don't want cache.
     */
    public AuthenticationContext(Context appContext, String authority,
            ITokenCacheStore tokenCacheStore) {
        initialize(appContext, authority, tokenCacheStore, true, false);
    }

    private void initialize(Context appContext, String authority, ITokenCacheStore tokenCacheStore,
            boolean validateAuthority, boolean defaultCache) {
        if (appContext == null) {
            throw new IllegalArgumentException("appContext");
        }
        if (authority == null) {
            throw new IllegalArgumentException("authority");
        }
        mBrokerProxy = new BrokerProxy(appContext);
        if (!defaultCache && !mBrokerProxy.canUseLocalCache()) {
            throw new UnsupportedOperationException("Local cache is not supported for broker usage");
        }
        mContext = appContext;
        checkInternetPermission();
        mAuthority = extractAuthority(authority);
        mValidateAuthority = validateAuthority;
        mTokenCacheStore = tokenCacheStore;
    }

    /**
     * Returns referenced cache. You can use default cache, which uses
     * SharedPreferences and handles synchronization by itself.
     * 
     * @return ITokenCacheStore Current cache used
     */
    public ITokenCacheStore getCache() {

        return mTokenCacheStore;
    }

    /**
     * Gets if the ExtendedLifetime mode is enabled.
     *
     * @return True when ExtendedLifetime mode is enabled
     */
    public boolean getExtendedLifetimeEnabled() {
        return mExtendedLifetimeEnabled;
    }

    /**
     * The client need to set the flag to true explicitly to enable the ExtendedLifetime mode
     * The default value of flag is false.
     * ADAL will return the stale token when ExtendedLifetime mode is enabled and the server is down
     *
     * @param extendedLifetimeEnabled true if the ExtendedLifetime mode is on, false otherwise
     */
    public void setExtendedLifetimeEnabled(final boolean extendedLifetimeEnabled) {
        mExtendedLifetimeEnabled = extendedLifetimeEnabled;
    }

    /**
     * Gets authority that is used for this object of AuthenticationContext.
     * 
     * @return Authority
     */
    public String getAuthority() {
        return mAuthority;
    }

    /**
     * @return True If developer turn on the authority validation, false otherwise.
     */
    public boolean getValidateAuthority() {
        return mValidateAuthority;
    }

    /**
     * Gets username for current broker user.
     * 
     * @return Username
     */
    public String getBrokerUser() {
        if (mBrokerProxy != null) {
            return mBrokerProxy.getCurrentUser();
        }

        return null;
    }

    /**
     * Gets user info from broker. This should not be called on main thread.
     * 
     * @return An array of {@link UserInfo} that haven been authenticated via broker(can be null).
     * 
     * @throws IOException if the broker returned an error response that indicates that it encountered an IOException
     * while communicating with the authentication server.
     * @throws AuthenticatorException if there was an error communicating with the authenticator or if the
     * authenticator returned an invalid response.
     * @throws OperationCanceledException if the request was canceled for any reason.
     */
    public UserInfo[] getBrokerUsers() throws OperationCanceledException, AuthenticatorException,
            IOException {
        return mBrokerProxy != null ? mBrokerProxy.getBrokerUsers() : null;
    }

    /**
     * Get expected redirect Uri for your app to use in broker. You need to
     * register this redirectUri in order to get token from Broker.
     * 
     * @return RedirectUri string to use for broker requests.
     */
    public String getRedirectUriForBroker() {
        final PackageHelper packageHelper = new PackageHelper(mContext);
        final String packageName = mContext.getPackageName();

        // First available signature. Applications can be signed with multiple
        // signatures.
        final String signatureDigest = packageHelper.getCurrentSignatureForPackage(packageName);
        final String redirectUri = PackageHelper.getBrokerRedirectUrl(packageName, signatureDigest);
        Logger.v(TAG, "Broker redirectUri:" + redirectUri + " packagename:" + packageName
                + " signatureDigest:" + signatureDigest);
        return redirectUri;
    }

    /**
     * acquireToken will start interactive flow if needed. It checks the cache
     * to return existing result if not expired. It tries to use refresh token
     * if available. If it fails to get token with refresh token, it will remove
     * this refresh token from cache and start authentication.
     * 
     * @param activity required to launch authentication activity.
     * @param resource required resource identifier.
     * @param clientId required client identifier
     * @param redirectUri Optional. It will use package name info if not
     *            provided.
     * @param loginHint Optional login hint
     * @param callback required
     */
    public void acquireToken(Activity activity, String resource, String clientId,
            String redirectUri, String loginHint,
            AuthenticationCallback<AuthenticationResult> callback) {
        if (checkPreRequirements(resource, clientId, callback)) {
            redirectUri = getRedirectUri(redirectUri);

            final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                    clientId, redirectUri, loginHint, PromptBehavior.Auto, null,
                    getRequestCorrelationId(), getExtendedLifetimeEnabled());
            request.setUserIdentifierType(UserIdentifierType.LoginHint);
            createAcquireTokenRequest().acquireToken(wrapActivity(activity), false, request, callback);
        }
    }

    /**
     * acquireToken will start interactive flow if needed. It checks the cache
     * to return existing result if not expired. It tries to use the refresh
     * token if available. If it fails to get token with refresh token, it will
     * remove this refresh token from cache and fall back on the UI.
     * 
     * @param activity Calling activity
     * @param resource required resource identifier.
     * @param clientId required client identifier
     * @param redirectUri Optional. It will use packagename and provided suffix
     *            for this.
     * @param loginHint Optional. This parameter will be used to pre-populate
     *            the username field in the authentication form. Please note
     *            that the end user can still edit the username field and
     *            authenticate as a different user. This parameter can be null.
     * @param extraQueryParameters Optional. This parameter will be appended as
     *            is to the query string in the HTTP authentication request to
     *            the authority. The parameter can be null.
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     */
    public void acquireToken(Activity activity, String resource, String clientId,
            String redirectUri, String loginHint, String extraQueryParameters,
            AuthenticationCallback<AuthenticationResult> callback) {
        if (checkPreRequirements(resource, clientId, callback)) {
            redirectUri = getRedirectUri(redirectUri);

            final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                    clientId, redirectUri, loginHint, PromptBehavior.Auto, extraQueryParameters,
                    getRequestCorrelationId(), getExtendedLifetimeEnabled());
            request.setUserIdentifierType(UserIdentifierType.LoginHint);
            createAcquireTokenRequest().acquireToken(wrapActivity(activity), false, request, callback);
        }
    }

    /**
     * acquireToken will start interactive flow if needed. It checks the cache
     * to return existing result if not expired. It tries to use refresh token
     * if available. If it fails to get token with refresh token, behavior will
     * depend on options. If {@link PromptBehavior} is AUTO, it will remove this
     * refresh token from cache and fall back on the UI. Default is AUTO. if
     * {@link PromptBehavior} is Always, it will display prompt screen.
     * 
     * @param activity Calling activity
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param redirectUri Optional. It will use packagename and provided suffix
     *            for this.
     * @param prompt Optional. {@link PromptBehavior} added as query parameter
     *            to authorization url
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     */
    public void acquireToken(Activity activity, String resource, String clientId,
            String redirectUri, PromptBehavior prompt,
            AuthenticationCallback<AuthenticationResult> callback) {
        if (checkPreRequirements(resource, clientId, callback)) {
            redirectUri = getRedirectUri(redirectUri);

            final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                    clientId, redirectUri, null, prompt, null, getRequestCorrelationId(), getExtendedLifetimeEnabled());
            createAcquireTokenRequest().acquireToken(wrapActivity(activity), false, request, callback);
        }
    }

    /**
     * acquireToken will start interactive flow if needed. It checks the cache
     * to return existing result if not expired. It tries to use refresh token
     * if available. If it fails to get token with refresh token, behavior will
     * depend on options. If promptbehavior is AUTO, it will remove this refresh
     * token from cache and fall back on the UI if activitycontext is not null.
     * Default is AUTO.
     * 
     * @param activity Calling activity
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param redirectUri Optional. It will use packagename and provided suffix
     *            for this.
     * @param prompt Optional. added as query parameter to authorization url
     * @param extraQueryParameters Optional. added to authorization url
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     */
    public void acquireToken(Activity activity, String resource, String clientId,
            String redirectUri, PromptBehavior prompt, String extraQueryParameters,
            AuthenticationCallback<AuthenticationResult> callback) {
        if (checkPreRequirements(resource, clientId, callback)) {
            redirectUri = getRedirectUri(redirectUri);

            final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                    clientId, redirectUri, null, prompt, extraQueryParameters,
                    getRequestCorrelationId(), getExtendedLifetimeEnabled());
            createAcquireTokenRequest().acquireToken(wrapActivity(activity), false, request, callback);
        }
    }

    /**
     * acquireToken will start interactive flow if needed. It checks the cache
     * to return existing result if not expired. It tries to use refresh token
     * if available. If it fails to get token with refresh token, behavior will
     * depend on options. If promptbehavior is AUTO, it will remove this refresh
     * token from cache and fall back on the UI if activitycontext is not null.
     * Default is AUTO.
     * 
     * @param activity Calling activity
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param redirectUri Optional. It will use packagename and provided suffix
     *            for this.
     * @param loginHint Optional. It is used for cache and as a loginhint at
     *            authentication.
     * @param prompt Optional. added as query parameter to authorization url
     * @param extraQueryParameters Optional. added to authorization url
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     */
    public void acquireToken(Activity activity, String resource, String clientId,
            String redirectUri, String loginHint, PromptBehavior prompt,
            String extraQueryParameters, AuthenticationCallback<AuthenticationResult> callback) {
        if (checkPreRequirements(resource, clientId, callback)) {
            redirectUri = getRedirectUri(redirectUri);

            final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                    clientId, redirectUri, loginHint, prompt, extraQueryParameters,
                    getRequestCorrelationId(), getExtendedLifetimeEnabled());
            request.setUserIdentifierType(UserIdentifierType.LoginHint);
            createAcquireTokenRequest().acquireToken(wrapActivity(activity), false, request, callback);
        }
    }

    /**
     * It will start interactive flow if needed. It checks the cache to return
     * existing result if not expired. It tries to use refresh token if
     * available. If it fails to get token with refresh token, behavior will
     * depend on options. If promptbehavior is AUTO, it will remove this refresh
     * token from cache and fall back on the UI. Default is AUTO.
     * 
     * @param fragment It accepts both type of fragments.
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param redirectUri Optional. It will use packagename and provided suffix
     *            for this.
     * @param loginHint Optional. It is used for cache and as a loginhint at
     *            authentication.
     * @param prompt Optional. added as query parameter to authorization url
     * @param extraQueryParameters Optional. added to authorization url
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     */
    public void acquireToken(IWindowComponent fragment, String resource, String clientId,
            String redirectUri, String loginHint, PromptBehavior prompt,
            String extraQueryParameters, AuthenticationCallback<AuthenticationResult> callback) {
        if (checkPreRequirements(resource, clientId, callback)) {
            redirectUri = getRedirectUri(redirectUri);

            final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                    clientId, redirectUri, loginHint, prompt, extraQueryParameters,
                    getRequestCorrelationId(), getExtendedLifetimeEnabled());
            request.setUserIdentifierType(UserIdentifierType.LoginHint);
            createAcquireTokenRequest().acquireToken(fragment, false, request, callback);
        }
    }

    /**
     * This uses new dialog based prompt. It will create a handler to run the
     * dialog related code. It will start interactive flow if needed. It checks
     * the cache to return existing result if not expired. It tries to use
     * refresh token if available. If it fails to get token with refresh token,
     * behavior will depend on options. If promptbehavior is AUTO, it will
     * remove this refresh token from cache and fall back on the UI. Default is
     * AUTO.
     * 
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param redirectUri Optional. It will use packagename and provided suffix
     *            for this.
     * @param loginHint Optional. It is used for cache and as a loginhint at
     *            authentication.
     * @param prompt Optional. added as query parameter to authorization url
     * @param extraQueryParameters Optional. added to authorization url
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     */
    public void acquireToken(String resource, String clientId, String redirectUri,
            String loginHint, PromptBehavior prompt, String extraQueryParameters,
            AuthenticationCallback<AuthenticationResult> callback) {
        if (checkPreRequirements(resource, clientId, callback)) {
            redirectUri = getRedirectUri(redirectUri);

            final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                    clientId, redirectUri, loginHint, prompt, extraQueryParameters,
                    getRequestCorrelationId(), getExtendedLifetimeEnabled());
            request.setUserIdentifierType(UserIdentifierType.LoginHint);
            createAcquireTokenRequest().acquireToken(null, true, request, callback);
        }
    }

    /**
     * This is sync function. It will first look at the cache and automatically
     * checks for the token expiration. Additionally, if no suitable access
     * token is found in the cache, but refresh token is available, the function
     * will use the refresh token automatically. This method will not show UI
     * for the user. If prompt is needed, the method will return an exception
     * 
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param userId UserID obtained from
     *            {@link AuthenticationResult #getUserInfo()}
     * @return A {@link Future} object representing the
     *         {@link AuthenticationResult} of the call. It contains Access
     *         Token,the Access Token's expiration time, Refresh token, and
     *         {@link UserInfo}.
     * @throws AuthenticationException If silent request fails to get the token back.
     * @throws InterruptedException If the main thread is interrupted before or during the activity.
     */
    public AuthenticationResult acquireTokenSilentSync(String resource, String clientId, String userId)
            throws AuthenticationException, InterruptedException {
        final AtomicReference<AuthenticationResult> authenticationResult = new AtomicReference<>();
        final AtomicReference<Exception> exception = new AtomicReference<>();
        final CountDownLatch latch = new CountDownLatch(1);
        if (StringExtensions.isNullOrBlank(resource)) {
            throw new IllegalArgumentException("The required resource is null or blank.");
        }
        if (StringExtensions.isNullOrBlank(clientId)) {
            throw new IllegalArgumentException("The required clientId is null or blank.");
        }

        final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                clientId, userId, getRequestCorrelationId(), getExtendedLifetimeEnabled());
        request.setSilent(true);
        request.setPrompt(PromptBehavior.Auto);
        request.setUserIdentifierType(UserIdentifierType.UniqueId);

        final Looper currentLooper = Looper.myLooper();
        if (currentLooper != null && currentLooper == mContext.getMainLooper()) {
            Log.e(TAG, "Sync network calls must not be invoked in main thread. "
                    + "This method will throw android.os.NetworkOnMainThreadException in next major release",
                    new NetworkOnMainThreadException());
        }
        createAcquireTokenRequest().acquireToken(null, false, request,
                new AuthenticationCallback<AuthenticationResult>() {
                    @Override
                    public void onSuccess(AuthenticationResult result) {
                        authenticationResult.set(result);
                        latch.countDown();
                    }

                    @Override
                    public void onError(Exception exc) {
                        exception.set(exc);
                        latch.countDown();
                    }
        });

        latch.await();
        Exception e = exception.get();
        if (e != null) {
            if (e instanceof AuthenticationException) {
                throw (AuthenticationException) e;
            } else if (e instanceof RuntimeException) {
                throw (RuntimeException) e;
            }
            if (e.getCause() != null) {
                if (e.getCause() instanceof AuthenticationException) {
                    throw (AuthenticationException) e.getCause();
                } else if (e.getCause() instanceof RuntimeException) {
                    throw (RuntimeException) e.getCause();
                } else {
                    throw new AuthenticationException(ADALError.ERROR_SILENT_REQUEST, e.getCause()
                            .getMessage(), e.getCause());
                }
            }
            throw new AuthenticationException(ADALError.ERROR_SILENT_REQUEST, e.getMessage(), e);
        }

        return authenticationResult.get();
    }

    /**
     * The function will first look at the cache and automatically checks for
     * the token expiration. Additionally, if no suitable access token is found
     * in the cache, but refresh token is available, the function will use the
     * refresh token automatically. This method will not show UI for the user.
     * If prompt is needed, the method will return an exception
     * 
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param userId UserId obtained from {@link UserInfo} inside
     *            {@link AuthenticationResult}
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     * @return A {@link Future} object representing the
     *         {@link AuthenticationResult} of the call. It contains Access
     *         Token,the Access Token's expiration time, Refresh token, and
     *         {@link UserInfo}.
     * @deprecated Use the {@link #acquireTokenSilentAsync} method.
     */
    @Deprecated
    public Future<AuthenticationResult> acquireTokenSilent(String resource, String clientId,
            String userId, final AuthenticationCallback<AuthenticationResult> callback) {
        if (StringExtensions.isNullOrBlank(resource)) {
            throw new IllegalArgumentException("resource");
        }

        if (StringExtensions.isNullOrBlank(clientId)) {
            throw new IllegalArgumentException("clientId");
        }

        final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                clientId, userId, getRequestCorrelationId(), getExtendedLifetimeEnabled());
        request.setSilent(true);
        request.setPrompt(PromptBehavior.Auto);
        request.setUserIdentifierType(UserIdentifierType.UniqueId);
        final SettableFuture<AuthenticationResult> futureTask = new SettableFuture<AuthenticationResult>();
        createAcquireTokenRequest().acquireToken(null, false, request,
                new AuthenticationCallback<AuthenticationResult>() {
            @Override
            public void onSuccess(AuthenticationResult result) {
                if (callback != null) {
                    callback.onSuccess(result);
                }
                futureTask.set(result);
            }

            @Override
            public void onError(Exception exc) {
                if (callback != null) {
                    callback.onError(exc);
                }
                futureTask.setException(exc);
            }
        });
        return futureTask;
    }

    /**
     * The function will first look at the cache and automatically checks for
     * the token expiration. Additionally, if no suitable access token is found
     * in the cache, but refresh token is available, the function will use the
     * refresh token automatically. This method will not show UI for the user.
     * If prompt is needed, the method will return an exception
     *
     * @param resource required resource identifier.
     * @param clientId required client identifier.
     * @param userId UserId obtained from {@link UserInfo} inside
     *            {@link AuthenticationResult}
     * @param callback required {@link AuthenticationCallback} object for async
     *            call.
     */
    public void acquireTokenSilentAsync(String resource,
                                   String clientId,
                                   String userId,
                                   AuthenticationCallback<AuthenticationResult> callback) {
        if (StringExtensions.isNullOrBlank(resource)) {
            throw new IllegalArgumentException("resource");
        }

        if (StringExtensions.isNullOrBlank(clientId)) {
            throw new IllegalArgumentException("clientId");
        }

        if (callback == null) {
            throw new IllegalArgumentException("callback");
        }

        final AuthenticationRequest request = new AuthenticationRequest(mAuthority, resource,
                clientId, userId, getRequestCorrelationId(), getExtendedLifetimeEnabled());
        request.setSilent(true);
        request.setPrompt(PromptBehavior.Auto);
        request.setUserIdentifierType(UserIdentifierType.UniqueId);
        createAcquireTokenRequest().acquireToken(null, false, request, callback);
    }

    /**
     * acquire token using refresh token if cache is not used. Otherwise, use
     * acquireToken to let the ADAL handle the cache lookup and refresh token
     * request.
     * 
     * @param refreshToken Required.
     * @param clientId Required.
     * @param callback Required
     * @deprecated {@link AuthenticationContext#acquireTokenByRefreshToken(String, String, AuthenticationCallback)}
     * will be removed from the next major release. We don't recommend developers taking dependency on refresh token,
     * ADAL will handle the cache lookup, and refresh the RT if necessary.
     */
    @Deprecated
    public void acquireTokenByRefreshToken(String refreshToken, String clientId,
            AuthenticationCallback<AuthenticationResult> callback) {
        if (StringExtensions.isNullOrBlank(refreshToken)) {
            throw new IllegalArgumentException("Refresh token is not provided");
        }

        if (StringExtensions.isNullOrBlank(clientId)) {
            throw new IllegalArgumentException("ClientId is not provided");
        }

        if (callback == null) {
            throw new IllegalArgumentException("Callback is not provided");
        }

        final AuthenticationRequest request = new AuthenticationRequest(mAuthority,
                null, clientId, getRequestCorrelationId(), getExtendedLifetimeEnabled());

        // It is not using cache and refresh is not expected to
        // show authentication activity.
        request.setSilent(true);

        // Authenticator is not supported if user is managing the cache
        createAcquireTokenRequest().refreshTokenWithoutCache(refreshToken, request, callback);
    }

    /**
     * acquire token using refresh token if cache is not used. Otherwise, use
     * acquireToken to let the ADAL handle the cache lookup and refresh token
     * request.
     * 
     * @param refreshToken Required.
     * @param clientId Required.
     * @param resource Required resource identifier.
     * @param callback Required
     * @deprecated {@link AuthenticationContext#acquireTokenByRefreshToken(String, String, AuthenticationCallback)}
     * will be removed from the next major release. We don't recommend developers taking dependency on refresh token,
     * ADAL will handle the cache lookup, and refresh the RT if necessary.
     */
    @Deprecated
    public void acquireTokenByRefreshToken(String refreshToken, String clientId, String resource,
            AuthenticationCallback<AuthenticationResult> callback) {
        if (StringExtensions.isNullOrBlank(refreshToken)) {
            throw new IllegalArgumentException("Refresh token is not provided");
        }

        if (StringExtensions.isNullOrBlank(clientId)) {
            throw new IllegalArgumentException("ClientId is not provided");
        }

        if (callback == null) {
            throw new IllegalArgumentException("Callback is not provided");
        }

        // Authenticator is not supported if user is managing the cache
        final AuthenticationRequest request = new AuthenticationRequest(mAuthority,
                resource, clientId, getRequestCorrelationId(), getExtendedLifetimeEnabled());

        // It is not using cache and refresh is not expected to
        // show authentication activity.
        request.setSilent(true);
        createAcquireTokenRequest().refreshTokenWithoutCache(refreshToken, request, callback);
    }

    /**
     * This method wraps the implementation for onActivityResult at the related
     * Activity class. This method is called at UI thread.
     * 
     * @param requestCode Request code provided at the start of the activity.
     * @param resultCode Result code set from the activity.
     * @param data {@link Intent}
     */
    public void onActivityResult(final int requestCode, final int resultCode, final Intent data) {
        if (requestCode == AuthenticationConstants.UIRequest.BROWSER_FLOW) {
            final AcquireTokenRequest acquireTokenRequest = new AcquireTokenRequest(mContext, this);
            acquireTokenRequest.onActivityResult(requestCode, resultCode, data);
        }
    }

    /**
     * Active authentication activity can be cancelled if it exists. It may not
     * be cancelled if activity is not launched yet. RequestId is the hashcode
     * of your AuthenticationCallback.
     *
     * @param requestId Hash code value of your callback to cancel activity
     *            launch
     * @return true: if there is a valid waiting request and cancel message send
     *         successfully. false: Request does not exist or cancel message not
     *         send
     * @throws AuthenticationException if failed to get the waiting request
     */
    public boolean cancelAuthenticationActivity(final int requestId) throws AuthenticationException {
        final  AuthenticationRequestState waitingRequest = getWaitingRequest(requestId);

        if (waitingRequest == null || waitingRequest.getDelegate() == null) {
            // there is not any waiting callback
            Logger.v(TAG, "Current callback is empty. There is not any active authentication.");
            return true;
        }

        final String currentCorrelationInfo;
        if (waitingRequest.getRequest() != null) {
            currentCorrelationInfo = String.format(" CorrelationId: %s",
                    waitingRequest.getRequest().getCorrelationId().toString());
        } else {
            currentCorrelationInfo = "No correlation id associated with waiting request";
        }
        Logger.v(TAG, "Current callback is not empty. There is an active authentication Activity."
                + currentCorrelationInfo);

        // intent to cancel. Authentication activity registers for this message
        // at onCreate event.
        final Intent intent = new Intent(AuthenticationConstants.Browser.ACTION_CANCEL);
        final Bundle extras = new Bundle();
        intent.putExtras(extras);
        intent.putExtra(AuthenticationConstants.Browser.REQUEST_ID, requestId);
        // send intent to cancel any active authentication activity.
        // it may not cancel it, if activity takes some time to launch.

        final boolean cancelResult = LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
        if (cancelResult) {
            // clear callback if broadcast message was successful
            Logger.v(TAG, "Cancel broadcast message was successful." + currentCorrelationInfo);
            waitingRequest.setCancelled(true);
            waitingRequest.getDelegate().onError(new AuthenticationCancelError(
                    "Cancel broadcast message was successful."));
        } else {
            // Activity is not launched yet or receiver is not registered
            Logger.w(TAG, "Cancel broadcast message was not successful." + currentCorrelationInfo,
                    "", ADALError.BROADCAST_CANCEL_NOT_SUCCESSFUL);
        }

        return cancelResult;
    }

    /**
     * Get the CorrelationId set by user.
     *
     * @return UUID
     */
    public UUID getRequestCorrelationId() {
        if (mRequestCorrelationId == null) {
            return UUID.randomUUID();
        }

        return mRequestCorrelationId;
    }

    /**
     * Set CorrelationId to requests.
     *
     * @param requestCorrelationId The correlation id {@link UUID} to be set in the request.
     */
    public void setRequestCorrelationId(final UUID requestCorrelationId) {
        this.mRequestCorrelationId = requestCorrelationId;
        Logger.setCorrelationId(requestCorrelationId);
    }

    private IWindowComponent wrapActivity(final Activity activity) {
        if (activity == null) {
            throw new IllegalArgumentException("activity");
        }

        return new IWindowComponent() {
            private Activity mRefActivity = activity;

            @Override
            public void startActivityForResult(Intent intent, int requestCode) {
                // if user closed an app or switched to another activity
                // mRefActivity can die before this method got invoked
                if (mRefActivity != null) {
                    mRefActivity.startActivityForResult(intent, requestCode);
                }
            }
        };
    }

    private boolean checkPreRequirements(final String resource, final String clientId,
                                        final AuthenticationCallback<AuthenticationResult> callback) {
        //check the permissions required for the broker usage
        if (AuthenticationSettings.INSTANCE.getUseBroker()) {
            try {
                mBrokerProxy.verifyBrokerPermissionsAPI22AndLess();
            } catch (final UsageAuthenticationException exception) {
                callback.onError(exception);
                return false;
            }
        }

        if (mContext == null) {
            throw new IllegalArgumentException("context", new AuthenticationException(
                    ADALError.DEVELOPER_CONTEXT_IS_NOT_PROVIDED));
        }

        if (StringExtensions.isNullOrBlank(resource)) {
            throw new IllegalArgumentException("resource");
        }

        if (StringExtensions.isNullOrBlank(clientId)) {
            throw new IllegalArgumentException("clientId");
        }

        if (callback == null) {
            throw new IllegalArgumentException("callback");
        }

        return true;
    }
    
    private String getRedirectUri(String inputRedirectUri) {
        final String redirectUri;
        if (StringExtensions.isNullOrBlank(inputRedirectUri)) {
            redirectUri = mContext.getApplicationContext().getPackageName();
        } else {
            redirectUri = inputRedirectUri;
        }

        return redirectUri;
    }

    private AcquireTokenRequest createAcquireTokenRequest() {
        return new AcquireTokenRequest(mContext, this);
    }

    private static String extractAuthority(String authority) {
        if (!StringExtensions.isNullOrBlank(authority)) {

            // excluding the starting https:// or http://
            int thirdSlash = authority.indexOf('/', EXCLUDE_INDEX);

            // third slash is not the last character
            if (thirdSlash >= 0 && thirdSlash != (authority.length() - 1)) {
                int fourthSlash = authority.indexOf("/", thirdSlash + 1);
                if (fourthSlash < 0 || fourthSlash > thirdSlash + 1) {
                    if (fourthSlash >= 0) {
                        return authority.substring(0, fourthSlash);
                    }

                    return authority;
                }
            }
        }

        throw new IllegalArgumentException("authority");
    }

    private void checkInternetPermission() {
        PackageManager pm = mContext.getPackageManager();
        if (PackageManager.PERMISSION_GRANTED != pm.checkPermission("android.permission.INTERNET",
                mContext.getPackageName())) {
            throw new IllegalStateException(new AuthenticationException(ADALError.DEVELOPER_INTERNET_PERMISSION_MISSING));
        }
    }

    /**
     * Internal API of ADAL to serialize the family token cache item for the
     * given user.
     * 
     * Verify if the input uniqueUserId is valid and the broker is not used.
     * Then check if this user has family refresh token item in the cache. If
     * true, create an SSOStateContainer object with the family refresh token
     * item of this user and continue the serialization process.
     * 
     * @param uniqueUserId Unique user id used to lookup family token.
     * @return The serialized blob.
     * @throws AuthenticationException
     */
    String serialize(final String uniqueUserId) throws AuthenticationException {
        if (StringExtensions.isNullOrBlank(uniqueUserId)) {
            throw new IllegalArgumentException("uniqueUserId");
        }

        if (mBrokerProxy.canSwitchToBroker() != BrokerProxy.SwitchToBroker.CANNOT_SWITCH_TO_BROKER) {
            throw new UsageAuthenticationException(ADALError.FAIL_TO_EXPORT,
                    "Failed to export the family refresh token cache item because broker is enabled.");
        }

        /*
         * The current serialize/deserialize feature is for only supports MS
         * apps. So the client ID for the FoCI token cache item is hard coded
         * below.
         */
        final String cacheKey = CacheKey.createCacheKeyForFRT(this.getAuthority(),
                AuthenticationConstants.MS_FAMILY_ID, uniqueUserId);
        final TokenCacheItem tokenItem = this.getCache().getItem(cacheKey);

        if (tokenItem == null) {
            Logger.i(TAG, "Cannot find the family token cache item for this userID", "");
            throw new UsageAuthenticationException(ADALError.FAIL_TO_EXPORT,
                    "Failed to export the FID because no family token cache item is found.");
        }

        if (!StringExtensions.isNullOrBlank(tokenItem.getFamilyClientId())) {
            return SSOStateSerializer.serialize(tokenItem);
        } else {
            throw new IllegalArgumentException("tokenItem does not contain family refresh token");
        }
    }

    /**
     * Internal API of ADAL to provide the deserialization to the TokenCacheItem
     * 
     * The method will take the serializedBlob string as input and deserialize
     * the string into a tokenCacheItem. The deserialized tokenCacheItem will be
     * stored into the cache. Exceptions will be thrown for invalid input or the
     * broker is enabled.
     *
     * @param serializedBlob The blob to be deserialized.
     * @throws AuthenticationException
     */
    void deserialize(final String serializedBlob) throws AuthenticationException {
        if (StringExtensions.isNullOrBlank(serializedBlob)) {
            throw new IllegalArgumentException("serializedBlob");
        }

        if (mBrokerProxy.canSwitchToBroker() != BrokerProxy.SwitchToBroker.CANNOT_SWITCH_TO_BROKER) {
            throw new UsageAuthenticationException(ADALError.FAIL_TO_IMPORT, "Failed to import the serialized blob "
                    + "because broker is enabled.");
        }

        final TokenCacheItem tokenCacheItem = SSOStateSerializer.deserialize(serializedBlob);
        final String cacheKey = CacheKey.createCacheKey(tokenCacheItem);
        this.getCache().setItem(cacheKey, tokenCacheItem);  
    }

    void setIsAuthorityValidated(final boolean isAuthorityValidated) {
        mIsAuthorityValidated = isAuthorityValidated;
    }

    boolean getIsAuthorityValidated() {
        return mIsAuthorityValidated;
    }

    AuthenticationRequestState getWaitingRequest(final int requestId) throws AuthenticationException {
        Logger.v(TAG, "Get waiting request: " + requestId);
        AuthenticationRequestState request;

        synchronized (mDelegateMap) {
            request = mDelegateMap.get(requestId);
        }

        if (request == null) {
            Logger.e(TAG, "Request callback is not available for requestId:" + requestId,
                    "", ADALError.CALLBACK_IS_NOT_FOUND);
            throw new AuthenticationException(ADALError.CALLBACK_IS_NOT_FOUND,
                    "Request callback is not available for requestId:" + requestId);
        }

        return request;
    }

    void putWaitingRequest(final int requestId, final AuthenticationRequestState requestState) {
        if (requestState == null) {
            return;
        }
        
        Logger.v(TAG, "Put waiting request: " + requestId
                + getCorrelationInfoFromWaitingRequest(requestState));

        synchronized (mDelegateMap) {
            mDelegateMap.put(requestId, requestState);
        }
    }

    void removeWaitingRequest(int requestId) {
        Logger.v(TAG, "Remove waiting request: " + requestId);

        synchronized (mDelegateMap) {
            mDelegateMap.remove(requestId);
        }
    }

    /**
     * If request has correlationID, ADAL should report that instead of current
     * CorrelationId.
     */
    String getCorrelationInfoFromWaitingRequest(final AuthenticationRequestState waitingRequest) {
        UUID requestCorrelationID = getRequestCorrelationId();
        if (waitingRequest.getRequest() != null) {
            requestCorrelationID = waitingRequest.getRequest().getCorrelationId();
        }

        return String.format(" CorrelationId: %s", requestCorrelationID.toString());
    }

    /**
     * Version name for ADAL not for the app itself.
     *
     * @return The current SDK version.
     */
    public static String getVersionName() {
        // Package manager does not report for ADAL
        // AndroidManifest files are not merged, so it is returning hard coded
        // value
        return "1.11.0";
    }

    /**
     * A {@link Future}  whose result can be set by a {@link #set(Object)} or {@link #setException(Throwable)}.
     */
    static final class SettableFuture<V> extends FutureTask<V> {
        SettableFuture() {
            super(new Callable<V>() {
                @Override
                public V call() throws Exception {
                    return null;
                }
            });
        }
        
        @SuppressWarnings("PMD.UselessOverridingMethod")
        @Override
        public void set(V v) {
            super.set(v);
        }

        @SuppressWarnings("PMD.UselessOverridingMethod")
        @Override
        public void setException(Throwable t) {
            super.setException(t);
        }
    }
}
