/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.applicationinsights.internal.logger;

import com.microsoft.applicationinsights.core.dependencies.apachecommons.io.FileUtils;
import com.microsoft.applicationinsights.core.dependencies.apachecommons.io.FilenameUtils;
import com.microsoft.applicationinsights.core.dependencies.apachecommons.lang3.StringUtils;
import com.microsoft.applicationinsights.internal.logger.ConsoleLoggerOutput;
import com.microsoft.applicationinsights.internal.logger.DefaultLogFileProxyFactory;
import com.microsoft.applicationinsights.internal.logger.LocalFileSystemUtils;
import com.microsoft.applicationinsights.internal.logger.LogFileProxy;
import com.microsoft.applicationinsights.internal.logger.LogFileProxyFactory;
import com.microsoft.applicationinsights.internal.logger.LoggerOutput;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;

public final class FileLoggerOutput
implements LoggerOutput {
    private static final int MIN_SIZE_PER_LOG_FILE_IN_MB = 5;
    private static final int MAX_SIZE_PER_LOG_FILE_IN_MB = 500;
    private static final int MIN_NUMBER_OF_LOG_FILES = 2;
    private static final String SDK_LOGS_DEFAULT_FOLDER = "javasdklogs";
    private static final String LOG_FILE_SUFFIX_FOR_LISTING = "jsl";
    public static final String NUMBER_OF_FILES_ATTRIBUTE = "NumberOfFiles";
    public static final String TOTAL_SIZE_OF_LOG_FILES_IN_MB_ATTRIBUTE = "NumberOfTotalSizeInMB";
    public static final String LOG_FILES_BASE_FOLDER_PATH_ATTRIBUTE = "BaseFolderPath";
    public static final String UNIQUE_LOG_FILE_PREFIX_ATTRIBUTE = "UniquePrefix";
    private static final String DATE_FORMAT_NOW = "yyyy-MM-dd-HH-mm-ss";
    private String uniquePrefix;
    private LogFileProxy[] files;
    private int maxSizePerFileInMB;
    private int currentLogFileIndex;
    private File baseFolder;
    private LogFileProxyFactory factory;
    private SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");
    private ConsoleLoggerOutput fallbackLoggerOutput = new ConsoleLoggerOutput();

    public FileLoggerOutput(Map<String, String> loggerData) {
        this.uniquePrefix = loggerData.get(UNIQUE_LOG_FILE_PREFIX_ATTRIBUTE);
        if (StringUtils.isEmpty(this.uniquePrefix)) {
            throw new IllegalArgumentException(String.format("Unique log file prefix is not defined", new Object[0]));
        }
        this.uniquePrefix = this.uniquePrefix + '-';
        int numberOfFiles = this.getRequest(loggerData, NUMBER_OF_FILES_ATTRIBUTE, 2);
        int numberOfTotalMB = this.getRequest(loggerData, TOTAL_SIZE_OF_LOG_FILES_IN_MB_ATTRIBUTE, 5);
        String baseFolderPath = loggerData.get(LOG_FILES_BASE_FOLDER_PATH_ATTRIBUTE);
        this.factory = new DefaultLogFileProxyFactory();
        this.initialize(baseFolderPath, numberOfFiles, numberOfTotalMB);
    }

    private int getRequest(Map<String, String> loggerData, String requestName, int defaultValue) {
        int requestValue = defaultValue;
        String requestValueAsString = loggerData.get(requestName);
        if (StringUtils.isNotEmpty(requestValueAsString)) {
            try {
                requestValue = Integer.parseInt(loggerData.get(requestName));
            }
            catch (Exception e) {
                this.fallbackLoggerOutput.log(String.format("Error: invalid value '%s' for '%s', using default: %d", requestValueAsString, requestName, defaultValue));
            }
        }
        return requestValue;
    }

    private void initialize(String baseFolderPath, int numberOfFiles, int numberOfTotalMB) {
        Path logFilePath;
        this.currentLogFileIndex = 0;
        if (StringUtils.isEmpty(baseFolderPath)) {
            baseFolderPath = LocalFileSystemUtils.getTempDir().getAbsolutePath();
            logFilePath = Paths.get(baseFolderPath, SDK_LOGS_DEFAULT_FOLDER);
        } else {
            logFilePath = Paths.get(baseFolderPath, new String[0]);
        }
        if (numberOfFiles < 2) {
            numberOfFiles = 2;
        }
        this.files = new LogFileProxy[numberOfFiles];
        int tempSizePerFileInMB = numberOfTotalMB / numberOfFiles;
        if (tempSizePerFileInMB < 5) {
            tempSizePerFileInMB = 5;
        } else if (tempSizePerFileInMB > 500) {
            tempSizePerFileInMB = 500;
        }
        this.maxSizePerFileInMB = tempSizePerFileInMB;
        if (!Files.exists(logFilePath, new LinkOption[0])) {
            try {
                this.baseFolder = Files.createDirectories(logFilePath, new FileAttribute[0]).toFile();
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            this.baseFolder = logFilePath.toFile();
            this.attachToExisting();
        }
    }

    @Override
    public synchronized void log(String message) {
        try {
            LogFileProxy logFileProxy = this.getCurrentLogFileProxy();
            if (logFileProxy != null) {
                logFileProxy.writeLine(message);
            }
        }
        catch (IOException e) {
            this.fallbackLoggerOutput.log(String.format("Failed to write to log to file exception: %s. Message '%s'", e.toString(), message));
        }
    }

    @Override
    public void close() {
        LogFileProxy currentLogger = this.files[this.currentLogFileIndex];
        if (currentLogger != null) {
            try {
                this.files[this.currentLogFileIndex] = null;
                currentLogger.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public void setLogProxyFactory(LogFileProxyFactory factory) {
        this.factory = factory;
    }

    private LogFileProxy getCurrentLogFileProxy() throws IOException {
        LogFileProxy currentProxy = this.files[this.currentLogFileIndex];
        if (currentProxy != null && !currentProxy.isFull()) {
            return currentProxy;
        }
        return this.createNewFileProxy();
    }

    private LogFileProxy createNewFileProxy() throws IOException {
        LogFileProxy logFileProxy;
        LogFileProxy currentLogger = this.files[this.currentLogFileIndex];
        if (currentLogger != null) {
            try {
                currentLogger.close();
            }
            catch (IOException e) {
                this.fallbackLoggerOutput.log(String.format("Failed to close log file, exception: %s", e.toString()));
            }
        }
        ++this.currentLogFileIndex;
        if (this.currentLogFileIndex == this.files.length) {
            this.currentLogFileIndex = 0;
        }
        if ((currentLogger = this.files[this.currentLogFileIndex]) != null) {
            this.files[this.currentLogFileIndex] = null;
            try {
                currentLogger.delete();
            }
            catch (Exception e) {
                this.fallbackLoggerOutput.log(String.format("Failed to delete log file, exception: %s", e.toString()));
            }
        }
        Calendar cal = Calendar.getInstance();
        String filePrefix = this.uniquePrefix + this.simpleDateFormat.format(cal.getTime());
        this.files[this.currentLogFileIndex] = logFileProxy = this.factory.create(this.baseFolder, filePrefix, this.maxSizePerFileInMB);
        return logFileProxy;
    }

    private void attachToExisting() {
        try {
            List<FileAndDate> oldLogs = this.getExistingLogsFromNewToOld();
            this.attachToExisting(oldLogs);
        }
        catch (Exception e) {
            this.fallbackLoggerOutput.log(String.format("Failed to delete old log file, exception: %s", e.toString()));
        }
    }

    private void attachToExisting(List<FileAndDate> oldLogs) {
        if (oldLogs.isEmpty()) {
            return;
        }
        int filesIndex = this.currentLogFileIndex;
        int numberOfFilesFound = 0;
        for (FileAndDate oldLog : oldLogs) {
            try {
                if (numberOfFilesFound < this.files.length) {
                    LogFileProxy logFileProxy = this.factory.attach(oldLog.file, this.maxSizePerFileInMB);
                    if (logFileProxy == null) continue;
                    ++numberOfFilesFound;
                    this.files[filesIndex] = logFileProxy;
                    ++filesIndex;
                    continue;
                }
                Files.delete(oldLog.file.toPath());
            }
            catch (Exception e) {
                this.fallbackLoggerOutput.log(String.format("Failed to delete old log file: %s", e.toString()));
            }
        }
    }

    private List<FileAndDate> getExistingLogsFromNewToOld() {
        try {
            Collection<File> oldLogs = FileUtils.listFiles(this.baseFolder, new String[]{LOG_FILE_SUFFIX_FOR_LISTING}, false);
            ArrayList<File> asList = !(oldLogs instanceof List) ? new ArrayList<File>(oldLogs) : (ArrayList<File>)oldLogs;
            ArrayList<FileAndDate> filesByDate = new ArrayList<FileAndDate>();
            for (File file : asList) {
                Date fileDate = this.getFileDate(file);
                if (fileDate == null) continue;
                filesByDate.add(new FileAndDate(file, fileDate));
            }
            Collections.sort(filesByDate, new Comparator<FileAndDate>(){

                @Override
                public int compare(FileAndDate file1, FileAndDate file2) {
                    if (file1.date.before(file2.date)) {
                        return 1;
                    }
                    if (file2.date.before(file1.date)) {
                        return -1;
                    }
                    return 0;
                }
            });
            return filesByDate;
        }
        catch (Exception e) {
            return Collections.emptyList();
        }
    }

    private Date getFileDate(File file) {
        try {
            String fileName = FilenameUtils.getBaseName(file.getName());
            int index = fileName.indexOf(this.uniquePrefix);
            if (index != -1) {
                String dateString = fileName.substring(index + this.uniquePrefix.length(), index + this.uniquePrefix.length() + DATE_FORMAT_NOW.length());
                Date date = this.simpleDateFormat.parse(dateString);
                return date;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    private static class FileAndDate {
        public final File file;
        public final Date date;

        private FileAndDate(File file, Date date) {
            this.file = file;
            this.date = date;
        }
    }
}

