/*
 * Decompiled with CFR 0.152.
 */
package com.azure.data.cosmos.internal;

import com.azure.data.cosmos.BridgeInternal;
import com.azure.data.cosmos.CosmosClientException;
import com.azure.data.cosmos.CosmosResponseDiagnostics;
import com.azure.data.cosmos.RetryOptions;
import com.azure.data.cosmos.internal.Exceptions;
import com.azure.data.cosmos.internal.GlobalEndpointManager;
import com.azure.data.cosmos.internal.IDocumentClientRetryPolicy;
import com.azure.data.cosmos.internal.IRetryPolicy;
import com.azure.data.cosmos.internal.ResourceThrottleRetryPolicy;
import com.azure.data.cosmos.internal.RxDocumentServiceRequest;
import com.azure.data.cosmos.internal.Utils;
import com.azure.data.cosmos.internal.directconnectivity.WebExceptionUtility;
import java.net.URL;
import java.time.Duration;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.collections4.list.UnmodifiableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class ClientRetryPolicy
implements IDocumentClientRetryPolicy {
    private static final Logger logger = LoggerFactory.getLogger(ClientRetryPolicy.class);
    static final int RetryIntervalInMS = 1000;
    static final int MaxRetryCount = 120;
    private final IDocumentClientRetryPolicy throttlingRetry;
    private final GlobalEndpointManager globalEndpointManager;
    private final boolean enableEndpointDiscovery;
    private int failoverRetryCount;
    private int sessionTokenRetryCount;
    private boolean isReadRequest;
    private boolean canUseMultipleWriteLocations;
    private URL locationEndpoint;
    private RetryContext retryContext;
    private CosmosResponseDiagnostics cosmosResponseDiagnostics;
    private AtomicInteger cnt = new AtomicInteger(0);

    public ClientRetryPolicy(GlobalEndpointManager globalEndpointManager, boolean enableEndpointDiscovery, RetryOptions retryOptions) {
        this.throttlingRetry = new ResourceThrottleRetryPolicy(retryOptions.maxRetryAttemptsOnThrottledRequests(), retryOptions.maxRetryWaitTimeInSeconds());
        this.globalEndpointManager = globalEndpointManager;
        this.failoverRetryCount = 0;
        this.enableEndpointDiscovery = enableEndpointDiscovery;
        this.sessionTokenRetryCount = 0;
        this.canUseMultipleWriteLocations = false;
        this.cosmosResponseDiagnostics = BridgeInternal.createCosmosResponseDiagnostics();
    }

    @Override
    public Mono<IRetryPolicy.ShouldRetryResult> shouldRetry(Exception e) {
        logger.debug("retry count {}, isReadRequest {}, canUseMultipleWriteLocations {}, due to failure:", new Object[]{this.cnt.incrementAndGet(), this.isReadRequest, this.canUseMultipleWriteLocations, e});
        if (this.locationEndpoint == null) {
            logger.error("locationEndpoint is null because ClientRetryPolicy::onBeforeRequest(.) is not invoked, probably request creation failed due to invalid options, serialization setting, etc.");
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.error(e));
        }
        this.retryContext = null;
        CosmosClientException clientException = Utils.as(e, CosmosClientException.class);
        if (clientException != null && clientException.cosmosResponseDiagnostics() != null) {
            this.cosmosResponseDiagnostics = clientException.cosmosResponseDiagnostics();
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 3)) {
            logger.warn("Endpoint not writable. Will refresh cache and retry. ", (Throwable)e);
            return this.shouldRetryOnEndpointFailureAsync(false, true);
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 1008) && this.isReadRequest) {
            logger.warn("Endpoint not available for reads. Will refresh cache and retry. ", (Throwable)e);
            return this.shouldRetryOnEndpointFailureAsync(true, false);
        }
        if (WebExceptionUtility.isNetworkFailure(e)) {
            if (this.isReadRequest || WebExceptionUtility.isWebExceptionRetriable(e)) {
                logger.warn("Endpoint not reachable. Will refresh cache and retry. ", (Throwable)e);
                return this.shouldRetryOnEndpointFailureAsync(this.isReadRequest, false);
            }
            return this.shouldNotRetryOnEndpointFailureAsync(this.isReadRequest, false);
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 404) && Exceptions.isSubStatusCode(clientException, 1002)) {
            return Mono.just((Object)this.shouldRetryOnSessionNotAvailable());
        }
        return this.throttlingRetry.shouldRetry(e);
    }

    private IRetryPolicy.ShouldRetryResult shouldRetryOnSessionNotAvailable() {
        ++this.sessionTokenRetryCount;
        if (!this.enableEndpointDiscovery) {
            return IRetryPolicy.ShouldRetryResult.noRetry();
        }
        if (this.canUseMultipleWriteLocations) {
            UnmodifiableList<URL> endpoints;
            UnmodifiableList<URL> unmodifiableList = endpoints = this.isReadRequest ? this.globalEndpointManager.getReadEndpoints() : this.globalEndpointManager.getWriteEndpoints();
            if (this.sessionTokenRetryCount > endpoints.size()) {
                return IRetryPolicy.ShouldRetryResult.noRetry();
            }
            this.retryContext = new RetryContext(this.sessionTokenRetryCount - 1, this.sessionTokenRetryCount > 1);
            return IRetryPolicy.ShouldRetryResult.retryAfter(Duration.ZERO);
        }
        if (this.sessionTokenRetryCount > 1) {
            return IRetryPolicy.ShouldRetryResult.noRetry();
        }
        this.retryContext = new RetryContext(this.sessionTokenRetryCount - 1, false);
        return IRetryPolicy.ShouldRetryResult.retryAfter(Duration.ZERO);
    }

    private Mono<IRetryPolicy.ShouldRetryResult> shouldRetryOnEndpointFailureAsync(boolean isReadRequest, boolean forceRefresh) {
        if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
            logger.warn("ShouldRetryOnEndpointFailureAsync() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        Mono<Void> refreshLocationCompletable = this.refreshLocation(isReadRequest, forceRefresh);
        Duration retryDelay = Duration.ZERO;
        if (!this.isReadRequest) {
            logger.debug("Failover happening. retryCount {}", (Object)this.failoverRetryCount);
            if (this.failoverRetryCount > 1) {
                retryDelay = Duration.ofMillis(1000L);
            }
        } else {
            retryDelay = Duration.ofMillis(1000L);
        }
        return refreshLocationCompletable.then(Mono.just((Object)IRetryPolicy.ShouldRetryResult.retryAfter(retryDelay)));
    }

    private Mono<IRetryPolicy.ShouldRetryResult> shouldNotRetryOnEndpointFailureAsync(boolean isReadRequest, boolean forceRefresh) {
        if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
            logger.warn("ShouldRetryOnEndpointFailureAsync() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        Mono<Void> refreshLocationCompletable = this.refreshLocation(isReadRequest, forceRefresh);
        return refreshLocationCompletable.then(Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry()));
    }

    private Mono<Void> refreshLocation(boolean isReadRequest, boolean forceRefresh) {
        ++this.failoverRetryCount;
        if (isReadRequest) {
            logger.warn("marking the endpoint {} as unavailable for read", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForRead(this.locationEndpoint);
        } else {
            logger.warn("marking the endpoint {} as unavailable for write", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForWrite(this.locationEndpoint);
        }
        return this.globalEndpointManager.refreshLocationAsync(null, forceRefresh);
    }

    @Override
    public void onBeforeSendRequest(RxDocumentServiceRequest request) {
        this.isReadRequest = request.isReadOnlyRequest();
        this.canUseMultipleWriteLocations = this.globalEndpointManager.CanUseMultipleWriteLocations(request);
        if (request.requestContext != null) {
            request.requestContext.cosmosResponseDiagnostics = this.cosmosResponseDiagnostics;
        }
        if (request.requestContext != null) {
            request.requestContext.ClearRouteToLocation();
        }
        if (this.retryContext != null) {
            request.requestContext.RouteToLocation(this.retryContext.retryCount, this.retryContext.retryRequestOnPreferredLocations);
        }
        this.locationEndpoint = this.globalEndpointManager.resolveServiceEndpoint(request);
        if (request.requestContext != null) {
            request.requestContext.RouteToLocation(this.locationEndpoint);
        }
    }

    private class RetryContext {
        public int retryCount;
        public boolean retryRequestOnPreferredLocations;

        public RetryContext(int retryCount, boolean retryRequestOnPreferredLocations) {
            this.retryCount = retryCount;
            this.retryRequestOnPreferredLocations = retryRequestOnPreferredLocations;
        }
    }
}

