package com.microsoft.azure.documentdb.internal;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.ThreadLocalRandom;

final class ReadTimeoutRetryPolicy {
    private final static Logger LOGGER = LoggerFactory.getLogger(ResourceThrottleRetryPolicy.class);

    private final int maxAttemptCount;
    private final int maxWaitTimeInMilliseconds;

    private volatile long retryAfterInMilliseconds = 0;
    private volatile int currentAttemptCount = 0;
    private volatile int cumulativeWaitTime = 0;

    public ReadTimeoutRetryPolicy(int maxAttemptCount, int maxWaitTimeInSeconds) {
        if (maxWaitTimeInSeconds > Integer.MAX_VALUE / 1000) {
            throw new IllegalArgumentException("maxWaitTimeInSeconds must be less than " + Integer.MAX_VALUE / 1000);
        }

        this.maxAttemptCount = maxAttemptCount;
        this.maxWaitTimeInMilliseconds = 1000 * maxWaitTimeInSeconds;
    }

    public long getRetryAfterInMilliseconds() {
        return this.retryAfterInMilliseconds;
    }

    /**
     * Should the caller retry the operation.
     * <p>
     * This retry policy should only be invoked if HttpStatusCode is 429 (Too Many Requests).
     *
     * @param exception the exception to check.
     * @return true if should retry.
     */
    public boolean shouldRetry(IllegalStateException exception) {
        this.retryAfterInMilliseconds = 0;

        if (this.currentAttemptCount >= this.maxAttemptCount) {
            LOGGER.warn("Operation will not retried because current attempt count of {} exceeded the max attempt count of {}",
                    this.currentAttemptCount,
                    this.maxAttemptCount,
                    exception);

            return false;
        }

        long maxRetryAfterInMilliseconds = this.currentAttemptCount * 100;
        if (maxRetryAfterInMilliseconds > 0) {
            this.retryAfterInMilliseconds = ThreadLocalRandom.current().nextLong(maxRetryAfterInMilliseconds);
        }

        this.cumulativeWaitTime += this.retryAfterInMilliseconds;
        if (this.cumulativeWaitTime >= this.maxWaitTimeInMilliseconds) {
            LOGGER.warn("Operation will not retried because current wait time of {} milliseconds exceeded the max wait time of {} milliseonds",
                    this.cumulativeWaitTime,
                    this.maxWaitTimeInMilliseconds,
                    exception);

            return false;
        }

        LOGGER.debug("Operation will be retried after {} milliseconds. Exception: {}",
                this.retryAfterInMilliseconds,
                exception.getMessage());
        this.currentAttemptCount++;
        return true;
    }
}
