package com.microsoft.azure.documentdb.internal;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.microsoft.azure.documentdb.DocumentClientException;

final class ResourceThrottleRetryPolicy implements RetryPolicy {
    private final static long defaultRetryInSeconds = 5;
    private final static Logger LOGGER = LoggerFactory.getLogger(ResourceThrottleRetryPolicy.class);

    private final int maxAttemptCount;
    private final int maxWaitTimeInMilliseconds;

    private volatile long retryAfterInMilliseconds = 0;
    private volatile int currentAttemptCount = 0;
    private volatile int cumulativeWaitTime = 0;

    public ResourceThrottleRetryPolicy(int maxAttemptCount, int maxWaitTimeInSeconds) {
        if (maxWaitTimeInSeconds > Integer.MAX_VALUE / 1000) {
            throw new IllegalArgumentException("maxWaitTimeInSeconds must be less than " + Integer.MAX_VALUE / 1000);
        }

        this.maxAttemptCount = maxAttemptCount;
        this.maxWaitTimeInMilliseconds = 1000 * maxWaitTimeInSeconds;
    }

    public long getRetryAfterInMilliseconds() {
        return this.retryAfterInMilliseconds;
    }

    /**
     * Should the caller retry the operation.
     * <p>
     * This retry policy should only be invoked if HttpStatusCode is 429 (Too Many Requests).
     *
     * @param exception the exception to check.
     * @return true if should retry.
     */
    public boolean shouldRetry(DocumentClientException exception) {
        this.retryAfterInMilliseconds = 0;

        if (this.currentAttemptCount >= this.maxAttemptCount) {
            return false;
        }

        this.retryAfterInMilliseconds =
                exception.getRetryAfterInMilliseconds();
        if (this.retryAfterInMilliseconds == 0) {
            // we should never reach here as BE should turn non-zero of
            // retry delay.
            this.retryAfterInMilliseconds = defaultRetryInSeconds * 1000;
        }

        this.cumulativeWaitTime += this.retryAfterInMilliseconds;
        if (this.cumulativeWaitTime >= this.maxWaitTimeInMilliseconds) {
            return false;
        }

        LOGGER.debug("Operation will be retried after {} milliseconds. Exception: {}",
                this.retryAfterInMilliseconds,
                exception.getMessage());
        this.currentAttemptCount++;
        return true;
    }
}
