package com.microsoft.azure.documentdb.internal.directconnectivity;

import java.io.IOException;
import java.net.SocketException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.NoHttpResponseException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.conn.HttpHostConnectException;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.microsoft.azure.documentdb.ConnectionPolicy;
import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.Error;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.ErrorUtils;
import com.microsoft.azure.documentdb.internal.HttpConstants;
import com.microsoft.azure.documentdb.internal.OperationType;
import com.microsoft.azure.documentdb.internal.ResourceType;
import com.microsoft.azure.documentdb.internal.RuntimeConstants;
import com.microsoft.azure.documentdb.internal.UserAgentContainer;
import com.microsoft.azure.documentdb.internal.Utils;

/**
 * Used internally to represents the transport client that communicates with the Azure Cosmos DB database service.
 */
public class HttpTransportClient extends TransportClient {

    // using multiple httpclient-connection pools to workaround the contention in
    // PoolingHttpClientConnectionManager
    private static final int HTTP_CLIENTS = Utils.getConcurrencyFactor();

    private final Logger logger = LoggerFactory.getLogger(HttpTransportClient.class);
    private static final UserAgentContainer defaultUserAgentContainer = new UserAgentContainer();

    private HttpClient[] httpClient;
    private HttpHost proxy;
    private boolean handleServiceUnavailableFromProxy;
    private final UserAgentContainer requestUserAgentContainer;

    public HttpTransportClient(HttpClient[] httpClients,
            ConnectionPolicy connectionPolicy,
            UserAgentContainer userAgentContainer) {
        this.httpClient = httpClients;
        this.proxy = connectionPolicy.getProxy();
        this.handleServiceUnavailableFromProxy = connectionPolicy.getHandleServiceUnavailableFromProxy();
        this.requestUserAgentContainer = userAgentContainer == null? HttpTransportClient.defaultUserAgentContainer : userAgentContainer;
    }

    public HttpTransportClient(ConnectionPolicy connectionPolicy,
            UserAgentContainer userAgentContainer) {
        this(HttpTransportClient.createHttpClients(connectionPolicy), connectionPolicy, userAgentContainer);
    }

    private static HttpClient[] createHttpClients(ConnectionPolicy connectionPolicy) {
        HttpClient[] httpClients = new HttpClient[HTTP_CLIENTS];
        for (int i = 0; i < httpClients.length; ++i) {
            PoolingHttpClientConnectionManager connectionManager = HttpClientFactory.createConnectionManager(
                    connectionPolicy.getMaxPoolSize(),
                    connectionPolicy.getIdleConnectionTimeout(),
                    connectionPolicy.getDirectRequestTimeout());
            httpClients[i] = HttpClientFactory.createHttpClient(connectionManager,
                    connectionPolicy.getDirectRequestTimeout());
        }
        return httpClients;
    }

    private static void addHeader(HttpRequestBase httpRequest, String headerName, DocumentServiceRequest request) {
        String headerValue = request.getHeaders().get(headerName);
        addHeader(httpRequest, headerName, headerValue);
    }

    private static void addHeader(HttpRequestBase httpRequest, String headerName, String headerValue) {
        if (!StringUtils.isEmpty(headerValue)) {
            httpRequest.addHeader(headerName, headerValue);
        }
    }

    private static String GetDateHeader(Map<String, String> headers) {
        if (headers == null) {
            return StringUtils.EMPTY;
        }

        // Since Date header is overridden by some proxies/http client libraries, we support
        // an additional date header 'x-ms-date' and prefer that to the regular 'date' header.
        String date = headers.get(HttpConstants.HttpHeaders.X_DATE);
        if (StringUtils.isEmpty(date)) {
            date = headers.get(HttpConstants.HttpHeaders.HTTP_DATE);
        }

        return date != null ? date : StringUtils.EMPTY;
    }

    private void  setDefaultHeaders(HttpRequestBase httpRequest) {
        addHeader(httpRequest, HttpConstants.HttpHeaders.VERSION, HttpConstants.Versions.CURRENT_VERSION);
        addHeader(httpRequest, HttpConstants.HttpHeaders.USER_AGENT, requestUserAgentContainer.getUserAgent());
        addHeader(httpRequest, HttpConstants.HttpHeaders.ACCEPT, RuntimeConstants.MediaTypes.JSON);
        addHeader(httpRequest, HttpConstants.HttpHeaders.CACHE_CONTROL,  HttpConstants.HeaderValues.NoCache);
    }

    public HttpClient getHttpClient(URI physicalAddress) {
        return this.httpClient[Math.abs(physicalAddress.hashCode() % this.httpClient.length)];
    }

    public StoreResponse invokeStore(URI physicalAddress, HttpRequestBase httpRequest,
            boolean isMedia, String activityId, boolean isReadOnlyRequest)
                    throws DocumentClientException {
        HttpResponse response;
        try {
            logger.trace("Sending request {}", physicalAddress);
            response = getHttpClient(physicalAddress).execute(httpRequest, HttpClientContext.create());
            return this.processResponse(physicalAddress.getPath(), isMedia, response, activityId);
        } catch (IOException e) {
            if (isReadOnlyRequest) {
                logger.debug("Received exception {} on read only request, " +
                        "sending request to {}, will re-resolve the address",
                        e.getMessage(), physicalAddress);
                throw new DocumentClientException(HttpStatus.SC_GONE, e);
            } else if (e instanceof NoHttpResponseException ||
                    e instanceof UnknownHostException ||
                    e instanceof HttpHostConnectException ||
                    e instanceof ConnectTimeoutException ||
                    e instanceof SocketException && e.getMessage().contains("Connection reset")) {
                logger.debug("Received retrieable exception {}, " +
                        "sending request to {}, will re-resolve the address",
                        e.getMessage(), physicalAddress);
                throw new DocumentClientException(HttpStatus.SC_GONE, e);
            } else {
                throw new DocumentClientException(HttpStatus.SC_SERVICE_UNAVAILABLE, e,
                        Collections.singletonMap(HttpConstants.HttpHeaders.WRITE_REQUEST_TRIGGER_ADDRESS_REFRESH,  "1"));
            }
        } catch (DocumentClientException e) {
            throw e;
        } catch (Exception e) {
            throw e;
        } finally {
            httpRequest.releaseConnection();
        }
    }

    @Override
    public StoreResponse invokeStore(URI physicalAddress, DocumentServiceRequest request) throws DocumentClientException {
        HttpRequestBase httpRequest = this.prepareHttpMessage(request.getActivityId(), physicalAddress,
                request.getOperationType(), request.getResourceType(), request);
        return this.invokeStore(physicalAddress, httpRequest, request.getIsMedia(), request.getActivityId(),
                request.isReadOnlyRequest());
    }

    private StoreResponse processResponse(String requestUri, boolean isMedia, HttpResponse response, String activityId)
            throws DocumentClientException {
        if (requestUri == null) {
            throw new IllegalArgumentException("requestUri");
        }

        if (response == null) {
            Map<String, String> headers = new HashMap<String, String>();
            headers.put(HttpConstants.HttpHeaders.ACTIVITY_ID, activityId);
            headers.put(HttpConstants.HttpHeaders.REQUEST_VALIDATION_FAILURE, "1");

            String errorBodyTemplate = "{'code':'%d', 'message':'Message: {\"Errors\":[\"%s\"]}'}";
            String errorBody = String.format(errorBodyTemplate, HttpStatus.SC_INTERNAL_SERVER_ERROR, "The backend response was not in the correct format.");
            throw new DocumentClientException(HttpStatus.SC_INTERNAL_SERVER_ERROR, new Error(errorBody), headers);
        } else if (this.handleServiceUnavailableFromProxy &&
                response.getStatusLine().getStatusCode() == HttpConstants.StatusCodes.SERVICE_UNAVAILABLE && 
                response.getHeaders(HttpConstants.HttpHeaders.SERVER_VERSION) == null) {
            logger.debug("Received retriable exception: Service Unavailable from proxy server is missing " +
                    HttpConstants.HttpHeaders.SERVER_VERSION + " header, indicating connectivity issues."
                            + " Handling as GONE exception. Will re-resolve the address and retry.");
            throw new DocumentClientException(HttpStatus.SC_GONE, new Exception());
        }

        // Successful request is when status code is < 300 or status code is 304 ( Not modified )
        if (response.getStatusLine().getStatusCode() < 300 || response.getStatusLine().getStatusCode() == 304) {
            return StoreResponse.fromHttpResponse(response, isMedia, null);
        } else {
            ErrorUtils.maybeThrowException(requestUri, response, false, this.logger);
            return null;
        }
    }

    public HttpRequestBase prepareHttpMessage(String activityId, URI physicalAddress, OperationType operationType,
            ResourceType resourceType, DocumentServiceRequest request) {

        URI resourceUri = null;
        try {
            resourceUri = new URI(
                    physicalAddress.getScheme(),
                    physicalAddress.getUserInfo(),
                    physicalAddress.getHost(),
                    physicalAddress.getPort(),
                    physicalAddress.getPath() + Utils.trimBeginingAndEndingSlashes(request.getPath()),
                    physicalAddress.getQuery(),
                    physicalAddress.getFragment()
                    );
        } catch (URISyntaxException e) {
            throw new IllegalStateException(e);
        }
        HttpRequestBase httpRequest;
        switch (operationType) {
        case Create:
        case Upsert:
        case ExecuteJavaScript:
            HttpPost postRequest = new HttpPost(resourceUri);
            postRequest.setEntity(request.getBody());

            httpRequest = postRequest;
            break;
        case Replace:
            HttpPut putRequest = new HttpPut(resourceUri);
            putRequest.setEntity(request.getBody());

            httpRequest = putRequest;
            break;
        case Delete:
            HttpDelete deleteRequest = new HttpDelete(resourceUri);

            httpRequest = deleteRequest;
            break;
        case Read:
        case ReadFeed:
            HttpGet getRequest = new HttpGet(resourceUri);
            httpRequest = getRequest;
            break;
        case Query:
        case SqlQuery:
            HttpPost queryPostRequest = new HttpPost(resourceUri);
            httpRequest = queryPostRequest;
            queryPostRequest.setEntity(request.getBody());
            HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.CONTENT_TYPE, RuntimeConstants.MediaTypes.QUERY_JSON);
            HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.IS_QUERY, Boolean.TRUE.toString());
            break;
        case Head:
        case HeadFeed:
            HttpHead headRequest = new HttpHead(resourceUri);
            httpRequest = headRequest;
            break;
        default:
            throw new IllegalStateException("Invalid operation type");
        }

        setDefaultHeaders(httpRequest);

        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.RESPONSE_CONTINUATION_TOKEN_LIMIT_IN_KB, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.VERSION, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.USER_AGENT, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.PAGE_SIZE, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.PRE_TRIGGER_INCLUDE, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.PRE_TRIGGER_EXCLUDE, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.POST_TRIGGER_INCLUDE, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.POST_TRIGGER_EXCLUDE, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.AUTHORIZATION, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.INDEXING_DIRECTIVE, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.CONSISTENCY_LEVEL, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.SESSION_TOKEN, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.PREFER, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.RESOURCE_TOKEN_EXPIRY, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.ENABLE_SCAN_IN_QUERY, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.EMIT_VERBOSE_TRACES_IN_QUERY, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.CONTINUATION, request.getContinuation());
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.ACTIVITY_ID, activityId);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.PARTITION_KEY, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.PARTITION_KEY_RANGE_ID, request);

        String dateHeader = HttpTransportClient.GetDateHeader(request.getHeaders());
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.X_DATE, dateHeader);
        HttpTransportClient.addHeader(httpRequest, "Match", this.GetMatch(request, request.getOperationType()));

        String fanoutRequestHeader = request.getHeaders().get(WFConstants.BackendHeaders.IS_FANOUT_REQUEST);
        HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.IS_FANOUT_REQUEST, fanoutRequestHeader);

        if (!request.getIsNameBased()) {
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.ResourceId, request.getResourceId());
        }

        if (request.getResourceType() == ResourceType.DocumentCollection) {
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.COLLECTION_PARTITION_INDEX, request.getHeaders().get(WFConstants.BackendHeaders.COLLECTION_PARTITION_INDEX));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.COLLECTION_SERVICE_INDEX, request.getHeaders().get(WFConstants.BackendHeaders.COLLECTION_SERVICE_INDEX));
        }

        if (request.getHeaders().get(WFConstants.BackendHeaders.BIND_REPLICA_DIRECTIVE) != null) {
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.BIND_REPLICA_DIRECTIVE, request.getHeaders().get(WFConstants.BackendHeaders.BIND_REPLICA_DIRECTIVE));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.PRIMARY_MASTER_KEY, request.getHeaders().get(WFConstants.BackendHeaders.PRIMARY_MASTER_KEY));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.SECONDARY_MASTER_KEY, request.getHeaders().get(WFConstants.BackendHeaders.SECONDARY_MASTER_KEY));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.PRIMARY_READONLY_KEY, request.getHeaders().get(WFConstants.BackendHeaders.PRIMARY_READONLY_KEY));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.SECONDARY_READONLY_KEY, request.getHeaders().get(WFConstants.BackendHeaders.SECONDARY_READONLY_KEY));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.BIND_MIN_EFFECTIVE_PARTITION_KEY, request.getHeaders().get(WFConstants.BackendHeaders.BIND_MIN_EFFECTIVE_PARTITION_KEY));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.BIND_MAX_EFFECTIVE_PARTITION_KEY, request.getHeaders().get(WFConstants.BackendHeaders.BIND_MAX_EFFECTIVE_PARTITION_KEY));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.BIND_PARTITION_KEY_RANGE_ID, request.getHeaders().get(WFConstants.BackendHeaders.BIND_PARTITION_KEY_RANGE_ID));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.BIND_PARTITION_KEY_RANGE_RID_PREFIX, request.getHeaders().get(WFConstants.BackendHeaders.BIND_PARTITION_KEY_RANGE_RID_PREFIX));
            HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.MINIMUM_ALLOWED_CLIENT_VERSION, request.getHeaders().get(WFConstants.BackendHeaders.MINIMUM_ALLOWED_CLIENT_VERSION));
        }

        // Upsert
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.IS_UPSERT, request);

        // Allow Tentative Writes
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.ALLOW_TENTATIVE_WRITES, request);
        
        // SupportSpatialLegacyCoordinates
        HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.PARTITION_COUNT, request);

        HttpTransportClient.addHeader(httpRequest, WFConstants.BackendHeaders.COLLECTION_RID, request);

        // Stored procedure logging
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.SCRIPT_ENABLE_LOGGING, request);

        // Collection quota
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.POPULATE_QUOTA_INFO, request);

        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.DISABLE_RU_PER_MINUTE_USAGE, request);

        // ChangeFeed
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.A_IM, request);
        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.IF_MODIFIED_SINCE, request);

        HttpTransportClient.addHeader(httpRequest, HttpConstants.HttpHeaders.EXCLUDE_SYSTEM_PROPERTIES, request);

        if (this.proxy != null) {
            RequestConfig requestConfig = RequestConfig.custom().setProxy(this.proxy).build();
            httpRequest.setConfig(requestConfig);
        }

        httpRequest.setProtocolVersion(HttpVersion.HTTP_1_1);

        return httpRequest;
    }

    private String GetMatch(DocumentServiceRequest request, OperationType resourceOperation) {
        switch (resourceOperation) {
        case Delete:
        case ExecuteJavaScript:
        case Replace:
        case Update:
        case Upsert:
            return request.getHeaders().get(HttpConstants.HttpHeaders.IF_MATCH);

        case Read:
        case ReadFeed:
            return request.getHeaders().get(HttpConstants.HttpHeaders.IF_NONE_MATCH);

        default:
            return null;
        }
    }
}
