package com.microsoft.azure.documentdb.internal.directconnectivity;

import org.apache.http.HttpStatus;

import java.net.URI;

import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.internal.HttpConstants;
import com.microsoft.azure.documentdb.internal.RequestChargeTracker;
import com.microsoft.azure.documentdb.internal.VectorSessionToken;

/**
 * Used internally to represent a result of a read request from the store.
 */
public class StoreReadResult {
    private StoreResponse storeResponse;
    private DocumentClientException exception;
    private long LSN;
    VectorSessionToken sessionToken;

    private String partitionKeyRangeId;
    private long quorumAckedLSN;
    private long globalCommittedLSN;
    private long itemLSN;
    private long numberOfReadRegions;
    private double requestCharge;
    private int currentReplicaSetSize;
    private int currentWriteQuorum;
    private boolean isValid;
    private boolean isGoneException;
    private boolean isNotFoundException;
    private URI storePhysicalAddress;

    public StoreReadResult(StoreResponse storeResponse, DocumentClientException exception, long lSN,
                           String partitionKeyRangeId, long quorumAckedLSN, double requestCharge, int currentReplicaSetSize,
                           int currentWriteQuorum, boolean isValid, URI storePhysicalAddress, long globalCommittedLSN, long numberOfReadRegions,
                           long itemLSN, VectorSessionToken sessionToken) {
        this.storeResponse = storeResponse;
        this.exception = exception;
        this.LSN = lSN;
        this.partitionKeyRangeId = partitionKeyRangeId;
        this.quorumAckedLSN = quorumAckedLSN;
        this.requestCharge = requestCharge;
        this.currentReplicaSetSize = currentReplicaSetSize;
        this.currentWriteQuorum = currentWriteQuorum;
        this.isValid = isValid;
        this.isGoneException = this.exception != null && this.exception.getStatusCode() == HttpStatus.SC_GONE;
        this.isNotFoundException = this.exception != null && this.exception.getStatusCode() == HttpStatus.SC_NOT_FOUND;
        this.storePhysicalAddress = storePhysicalAddress;
        this.globalCommittedLSN = globalCommittedLSN;
        this.numberOfReadRegions = numberOfReadRegions;
        this.itemLSN = itemLSN;
        this.sessionToken = sessionToken;
    }

    public DocumentClientException getException() {
        return exception;
    }

    public StoreResponse getStoreResponse() {
        return storeResponse;
    }

    public long getLSN() {
        return LSN;
    }

    public void setLSN(long lSN) {
        LSN = lSN;
    }

    public VectorSessionToken getSessionToken() {
        return this.sessionToken;
    }
    
    public void setSessionToken(VectorSessionToken sessionToken) {
        this.sessionToken = sessionToken;
    }
    
    public long getGlobalCommittedLSN() {
        return globalCommittedLSN;
    }

    private void setGlobalCommittedLSN(long globalCommittedLSN) {
        this.globalCommittedLSN = globalCommittedLSN;
    }

    public long getItemLSN() {
        return itemLSN;
    }

    private void setItemLSN(long itemLSN) {
        this.itemLSN = itemLSN;
    }

    public long getNumberOfReadRegions() {
        return numberOfReadRegions;
    }

    private void setNumberOfReadRegions(long numberOfReadRegions) {
        this.numberOfReadRegions = numberOfReadRegions;
    }

    public String getPartitionKeyRangeId() {
        return partitionKeyRangeId;
    }

    public void setPartitionKeyRangeId(String partitionKeyRangeId) {
        this.partitionKeyRangeId = partitionKeyRangeId;
    }

    public long getQuorumAckedLSN() {
        return quorumAckedLSN;
    }

    public void setQuorumAckedLSN(long quorumAckedLSN) {
        this.quorumAckedLSN = quorumAckedLSN;
    }

    public double getRequestCharge() {
        return requestCharge;
    }

    private void setRequestCharge(RequestChargeTracker chargeTracker) {
        if (this.exception != null) {
            this.exception.getResponseHeaders().put(HttpConstants.HttpHeaders.REQUEST_CHARGE, Double.toString(chargeTracker.getTotalRequestCharge()));
        } else if (this.storeResponse.getResponseHeaderNames() != null) {
            for (int i = 0; i < this.storeResponse.getResponseHeaderNames().length; i++) {
                if (this.storeResponse.getResponseHeaderNames()[i].equals(HttpConstants.HttpHeaders.REQUEST_CHARGE)) {
                    this.storeResponse.getResponseHeaderValues()[i] = Double.toString(chargeTracker.getTotalRequestCharge());
                }
            }
        }
    }

    public int getCurrentReplicaSetSize() {
        return currentReplicaSetSize;
    }

    public void setCurrentReplicaSetSize(int currentReplicaSetSize) {
        this.currentReplicaSetSize = currentReplicaSetSize;
    }

    public int getCurrentWriteQuorum() {
        return currentWriteQuorum;
    }

    public void setCurrentWriteQuorum(int currentWriteQuorum) {
        this.currentWriteQuorum = currentWriteQuorum;
    }

    public boolean isValid() {
        return isValid;
    }

    public void setValid(boolean isValid) {
        this.isValid = isValid;
    }

    public boolean isGoneException() {
        return isGoneException;
    }

    public void setGoneException(boolean isGoneException) {
        this.isGoneException = isGoneException;
    }

    public boolean isNotFoundException() {
        return isNotFoundException;
    }

    public void setNotFoundException(boolean isNotFoundException) {
        this.isNotFoundException = isNotFoundException;
    }

    public URI getStorePhysicalAddress() {
        return storePhysicalAddress;
    }

    public void setStorePhysicalAddress(URI storePhysicalAddress) {
        this.storePhysicalAddress = storePhysicalAddress;
    }

    public StoreResponse toStoreResponse(RequestChargeTracker chargeTracker) throws DocumentClientException {
        if (!this.isValid) {
            if (this.exception == null) {
                throw new DocumentClientException((int) HttpStatus.SC_INTERNAL_SERVER_ERROR, "Unknown server error occurred when processing this request.");
            }

            throw this.exception;
        }

        if (chargeTracker != null) {
            this.setRequestCharge(chargeTracker);
        }

        if (this.exception != null) throw this.exception;
        return this.storeResponse;
    }

    @Override
    public String toString() {
        int statusCode = 0;
        if (this.storeResponse != null) {
            statusCode = this.storeResponse.getStatus();
        } else if (this.exception != null) {
            statusCode = this.exception.getStatusCode();
        }

        return "StoreReadResult{" +
                "statusCode=" + statusCode +
                ", storePhysicalAddress=" + storePhysicalAddress +
                ", LSN=" + LSN +
                ", sessionToken=" + (sessionToken != null ? sessionToken.convertToString() : null) +
                ", partitionKeyRangeId='" + partitionKeyRangeId + '\'' +
                ", quorumAckedLSN=" + quorumAckedLSN +
                ", globalCommittedLSN=" + globalCommittedLSN +
                ", itemLSN=" + itemLSN +
                ", numberOfReadRegions=" + numberOfReadRegions +
                ", requestCharge=" + requestCharge +
                ", currentReplicaSetSize=" + currentReplicaSetSize +
                ", currentWriteQuorum=" + currentWriteQuorum +
                ", isValid=" + isValid +
                ", isGoneException=" + isGoneException +
                ", isNotFoundException=" + isNotFoundException +
                ", exception='" + (this.exception != null ? this.exception.getInnerErrorMessage() : null) + "\'" +
                '}';
    }
}
