package com.microsoft.azure.documentdb.internal.routing;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.PartitionKeyDefinition;
import com.microsoft.azure.documentdb.PartitionKeyRange;
import com.microsoft.azure.documentdb.SqlQuerySpec;
import com.microsoft.azure.documentdb.internal.HttpConstants;
import com.microsoft.azure.documentdb.internal.Utils;
import com.microsoft.azure.documentdb.internal.query.PartitionedQueryExecutionInfo;
import com.microsoft.azure.documentdb.internal.query.QueryPartitionProvider;

/**
 * Used internally to provide helper functions to work with partition routing in the Azure Cosmos DB database service.
 */
public class PartitionRoutingHelper {

    private final static Logger logger = LoggerFactory.getLogger(PartitionRoutingHelper.class);

    public static PartitionKeyRange tryGetTargetRangeFromContinuationTokenRange(
            List<Range<String>> providedPartitionKeyRanges,
            RoutingMapProvider routingMapProvider,
            String collectionSelfLink,
            Range<String> rangeFromContinuationToken) {

        // For queries such as "SELECT * FROM root WHERE false",
        // we will have empty ranges and just forward the request to the first partition
        if (providedPartitionKeyRanges.size() == 0) {
            return routingMapProvider.tryGetRangeByEffectivePartitionKey(collectionSelfLink,
                    PartitionKeyInternalHelper.MinimumInclusiveEffectivePartitionKey);
        }

        // Initially currentRange will be empty
        if (rangeFromContinuationToken.isEmpty()) {
            Range<String> minimumRange = PartitionRoutingHelper.min(providedPartitionKeyRanges, new Range.MinComparator<String>());

            return routingMapProvider.tryGetRangeByEffectivePartitionKey(collectionSelfLink, minimumRange.getMin());
        }

        PartitionKeyRange targetPartitionKeyRange = routingMapProvider.tryGetRangeByEffectivePartitionKey(
                collectionSelfLink, rangeFromContinuationToken.getMin()
                );

        if (targetPartitionKeyRange == null
                || !rangeFromContinuationToken.equals(targetPartitionKeyRange.toRange())) {
            // Cannot find target range. Either collection was resolved incorrectly or the range was split.
            // We cannot distinguish here. Returning null and refresh the cache and retry.
            return null;
        }

        return targetPartitionKeyRange;
    }

    private static <T> T min(List<T> values, Comparator<T> comparer) {
        if (values.size() == 0) {
            throw new IllegalArgumentException("values");
        }

        T min = values.get(0);
        for (int i = 1; i < values.size(); ++i) {
            if (comparer.compare(values.get(i), min) < 0) {
                min = values.get(i);
            }
        }

        return min;
    }

    public static Range<String> extractPartitionKeyRangeFromContinuationToken(Map<String, String> headers) {
        Range<String> range = Range.getEmptyRange(PartitionKeyInternalHelper.MinimumInclusiveEffectivePartitionKey);

        String continuationToken = headers.get(HttpConstants.HttpHeaders.CONTINUATION);

        if (continuationToken == null || continuationToken.isEmpty()) {
            return range;
        }

        CompositeContinuationToken compositeContinuationToken;
        try {
            compositeContinuationToken = Utils.getSimpleObjectMapper().readValue(continuationToken,
                    CompositeContinuationToken.class);
            if (compositeContinuationToken.getRange() != null) {
                range = compositeContinuationToken.getRange();
            }
            headers.put(HttpConstants.HttpHeaders.CONTINUATION,
                    compositeContinuationToken.getToken() != null && !compositeContinuationToken.getToken().isEmpty()
                    ? compositeContinuationToken.getToken()
                            : "");
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }

        return range;
    }

    public static boolean tryAddPartitionKeyRangeToContinuationToken(
            Map<String, String> headers,
            List<Range<String>> providedPartitionKeyRanges,
            RoutingMapProvider routingMapProvider,
            String collectionSelfLink,
            PartitionKeyRange currentRange) {
        PartitionKeyRange rangeToUse = currentRange;

        // We only need to get the next range if we have to
        if (headers.get(HttpConstants.HttpHeaders.CONTINUATION) == null
                || headers.get(HttpConstants.HttpHeaders.CONTINUATION).isEmpty()) {
            Range<String> nextProvidedRange = PartitionRoutingHelper.minAfter(
                    providedPartitionKeyRanges,
                    currentRange.toRange(),
                    new Range.MaxComparator<String>()
                    );

            if (nextProvidedRange == null) {
                return true;
            }

            String max = nextProvidedRange.getMin().compareTo(currentRange.getMaxExclusive()) > 0 ? nextProvidedRange.getMin()
                    : currentRange.getMaxExclusive();

            if (max.compareTo(PartitionKeyInternalHelper.MaximumExclusiveEffectivePartitionKey) == 0) {
                return true;
            }

            PartitionKeyRange nextRange = routingMapProvider.tryGetRangeByEffectivePartitionKey(collectionSelfLink, max);
            if (nextRange == null) {
                return false;
            }

            rangeToUse = nextRange;
        }

        if (rangeToUse != null) {
            headers.put(HttpConstants.HttpHeaders.CONTINUATION, PartitionRoutingHelper.addPartitionKeyRangeToContinuationToken(
                    headers.get(HttpConstants.HttpHeaders.CONTINUATION),
                    rangeToUse));
        }

        return true;
    }

    private static <T> T minAfter(List<T> values, T minValue, Comparator<T> comparer) {
        if (values.size() == 0) {
            throw new IllegalArgumentException("values");
        }

        T min = null;
        for (T value : values) {
            if (comparer.compare(value, minValue) > 0 && (min == null || comparer.compare(value, min) < 0)) {
                min = value;
            }
        }

        return min;
    }

    private static String addPartitionKeyRangeToContinuationToken(String continuationToken, PartitionKeyRange partitionKeyRange) {
        CompositeContinuationToken compositeContinuationToken = new CompositeContinuationToken();
        compositeContinuationToken.setToken(continuationToken);
        compositeContinuationToken.setRange(partitionKeyRange.toRange());
        try {
            String serializedCompositeToken = Utils.getSimpleObjectMapper().writeValueAsString(compositeContinuationToken);
            return serializedCompositeToken;
        } catch (JsonProcessingException e) {
            throw new IllegalStateException(e);
        }
    }

    public static List<Range<String>> getProvidedPartitionKeyRanges(
            SqlQuerySpec querySpec,
            boolean enableCrossPartitionQuery,
            boolean parallelizeCrossPartitionQuery,
            PartitionKeyDefinition partitionKeyDefinition,
            QueryPartitionProvider queryPartitionProvider,
            String clientApiVersion) throws DocumentClientException {
        if (querySpec == null) {
            throw new IllegalArgumentException("querySpec");
        }

        if (queryPartitionProvider == null) {
            throw new IllegalArgumentException("queryPartitionProvider");
        }

        if (partitionKeyDefinition != null && partitionKeyDefinition.getPaths().size() > 0) {
            PartitionedQueryExecutionInfo queryExecutionInfo = null;

            queryExecutionInfo = queryPartitionProvider.getPartitionQueryExcecutionInfo(querySpec, partitionKeyDefinition);

            if (queryExecutionInfo == null
                    || queryExecutionInfo.getQueryRanges() == null
                    || queryExecutionInfo.getQueryInfo() == null) {
                logger.debug("QueryPartitionProvider");
            }

            boolean isSinglePartitionQuery = queryExecutionInfo.getQueryRanges().size() == 1
                    && queryExecutionInfo.getQueryRanges().iterator().next().isSingleValue();
            if (!isSinglePartitionQuery) {
                if (!enableCrossPartitionQuery) {
                    throw new DocumentClientException(
                            HttpConstants.StatusCodes.BADREQUEST,
                            "Cross partition query is required but disabled. Please set x-ms-documentdb-query-enablecrosspartition " +
                            "to true, specify x-ms-documentdb-partitionkey, or revise your query to avoid this exception.");
                } else {
                    if (parallelizeCrossPartitionQuery
                            || (queryExecutionInfo.getQueryInfo() != null
                            && (queryExecutionInfo.getQueryInfo().hasTop()
                                    || queryExecutionInfo.getQueryInfo().hasOrderBy()
                                    || queryExecutionInfo.getQueryInfo().hasAggregates()))) {
                        throw new DocumentClientException(HttpConstants.StatusCodes.BADREQUEST,
                                "Cross partition query with TOP and/or ORDER BY is not supported. " + queryExecutionInfo.toJson()
                                );
                    }
                }
            }

            return queryExecutionInfo.getQueryRanges();
        } else {
            return new ArrayList<Range<String>>() {{
                add(Range.getPointRange(PartitionKeyInternalHelper.MinimumInclusiveEffectivePartitionKey));
            }};
        }
    }
}

final class CompositeContinuationToken {
    private String token;
    private Range<String> range;

    public String getToken() {
        return token;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public Range<String> getRange() {
        return range;
    }

    public void setRange(Range<String> range) {
        this.range = range;
    }
}