/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.eventprocessorhost;

import com.microsoft.azure.eventprocessorhost.EventProcessorHost;
import com.microsoft.azure.eventprocessorhost.ILeaseManager;
import com.microsoft.azure.eventprocessorhost.Lease;
import com.microsoft.azure.eventprocessorhost.LoggingUtils;
import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryLeaseManager
implements ILeaseManager {
    private EventProcessorHost host;
    private ExecutorService executor = Executors.newCachedThreadPool();
    private static final Logger TRACE_LOGGER = LoggerFactory.getLogger(InMemoryLeaseManager.class);

    public void initialize(EventProcessorHost host) {
        this.host = host;
    }

    @Override
    public int getLeaseRenewIntervalInMilliseconds() {
        return this.host.getPartitionManagerOptions().getLeaseRenewIntervalInSeconds() * 1000;
    }

    @Override
    public int getLeaseDurationInMilliseconds() {
        return this.host.getPartitionManagerOptions().getLeaseDurationInSeconds() * 1000;
    }

    @Override
    public Future<Boolean> leaseStoreExists() {
        return this.executor.submit(() -> this.leaseStoreExistsSync());
    }

    private Boolean leaseStoreExistsSync() {
        return InMemoryLeaseStore.singleton.existsMap();
    }

    @Override
    public Future<Boolean> createLeaseStoreIfNotExists() {
        return this.executor.submit(() -> this.createLeaseStoreIfNotExistsSync());
    }

    private Boolean createLeaseStoreIfNotExistsSync() {
        InMemoryLeaseStore.singleton.initializeMap(this.getLeaseDurationInMilliseconds());
        return true;
    }

    @Override
    public Future<Boolean> deleteLeaseStore() {
        return this.executor.submit(() -> this.deleteLeaseStoreSync());
    }

    private Boolean deleteLeaseStoreSync() {
        InMemoryLeaseStore.singleton.deleteMap();
        return true;
    }

    @Override
    public Future<Lease> getLease(String partitionId) {
        return this.executor.submit(() -> this.getLeaseSync(partitionId));
    }

    private InMemoryLease getLeaseSync(String partitionId) {
        InMemoryLease returnLease = null;
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(partitionId);
        if (leaseInStore == null) {
            TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), partitionId, "getLease() no existing lease"));
            returnLease = null;
        } else {
            returnLease = new InMemoryLease(leaseInStore);
        }
        return returnLease;
    }

    @Override
    public Iterable<Future<Lease>> getAllLeases() throws Exception {
        String[] partitionIds;
        ArrayList<Future<Lease>> leases = new ArrayList<Future<Lease>>();
        for (String id : partitionIds = this.host.getPartitionManager().getPartitionIds()) {
            leases.add(this.getLease(id));
        }
        return leases;
    }

    @Override
    public Future<Lease> createLeaseIfNotExists(String partitionId) {
        return this.executor.submit(() -> this.createLeaseIfNotExistsSync(partitionId));
    }

    private InMemoryLease createLeaseIfNotExistsSync(String partitionId) {
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(partitionId);
        InMemoryLease returnLease = null;
        if (leaseInStore != null) {
            TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), partitionId, "createLeaseIfNotExists() found existing lease, OK"));
            returnLease = new InMemoryLease(leaseInStore);
        } else {
            TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), partitionId, "createLeaseIfNotExists() creating new lease"));
            InMemoryLease newStoreLease = new InMemoryLease(partitionId);
            newStoreLease.setEpoch(0L);
            newStoreLease.setOwner("");
            InMemoryLeaseStore.singleton.setOrReplaceLease(newStoreLease);
            returnLease = new InMemoryLease(newStoreLease);
        }
        return returnLease;
    }

    @Override
    public Future<Void> deleteLease(Lease lease) {
        return this.executor.submit(() -> this.deleteLeaseSync(lease));
    }

    private Void deleteLeaseSync(Lease lease) {
        TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "Deleting lease"));
        InMemoryLeaseStore.singleton.removeLease((InMemoryLease)lease);
        return null;
    }

    @Override
    public Future<Boolean> acquireLease(Lease lease) {
        return this.executor.submit(() -> this.acquireLeaseSync((InMemoryLease)lease));
    }

    private Boolean acquireLeaseSync(InMemoryLease lease) {
        TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "Acquiring lease"));
        boolean retval = true;
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(lease.getPartitionId());
        if (leaseInStore != null) {
            InMemoryLease wasUnowned = InMemoryLeaseStore.singleton.atomicAquireUnowned(lease.getPartitionId(), this.host.getHostName());
            if (wasUnowned != null) {
                lease.setOwner(this.host.getHostName());
                TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "acquireLease() acquired lease"));
                leaseInStore = wasUnowned;
                lease.setExpirationTime(leaseInStore.getExpirationTime());
            } else {
                if (leaseInStore.getOwner().compareTo(this.host.getHostName()) == 0) {
                    TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "acquireLease() already hold lease"));
                } else {
                    String oldOwner = leaseInStore.getOwner();
                    leaseInStore.setOwner(this.host.getHostName());
                    lease.setOwner(this.host.getHostName());
                    TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "acquireLease() stole lease from " + oldOwner));
                }
                long newExpiration = System.currentTimeMillis() + (long)this.getLeaseDurationInMilliseconds();
                leaseInStore.setExpirationTime(newExpiration);
                lease.setExpirationTime(newExpiration);
            }
        } else {
            TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "acquireLease() can't find lease"));
            retval = false;
        }
        return retval;
    }

    @Override
    public Future<Boolean> renewLease(Lease lease) {
        return this.executor.submit(() -> this.renewLeaseSync((InMemoryLease)lease));
    }

    private Boolean renewLeaseSync(InMemoryLease lease) {
        TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "Renewing lease"));
        boolean retval = true;
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(lease.getPartitionId());
        if (leaseInStore != null) {
            if (leaseInStore.getOwner().compareTo(this.host.getHostName()) == 0) {
                long newExpiration = System.currentTimeMillis() + (long)this.getLeaseDurationInMilliseconds();
                leaseInStore.setExpirationTime(newExpiration);
                lease.setExpirationTime(newExpiration);
            } else {
                TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "renewLease() not renewed because we don't own lease"));
                retval = false;
            }
        } else {
            TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "renewLease() can't find lease"));
            retval = false;
        }
        return retval;
    }

    @Override
    public Future<Boolean> releaseLease(Lease lease) {
        return this.executor.submit(() -> this.releaseLeaseSync((InMemoryLease)lease));
    }

    private Boolean releaseLeaseSync(InMemoryLease lease) {
        TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "Releasing lease"));
        boolean retval = true;
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(lease.getPartitionId());
        if (leaseInStore != null) {
            if (!this.wrapIsExpired(leaseInStore) && leaseInStore.getOwner().compareTo(this.host.getHostName()) == 0) {
                TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "releaseLease() released OK"));
                leaseInStore.setOwner("");
                lease.setOwner("");
                leaseInStore.setExpirationTime(0L);
                lease.setExpirationTime(0L);
            } else {
                TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "releaseLease() not released because we don't own lease"));
                retval = false;
            }
        } else {
            TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "releaseLease() can't find lease"));
            retval = false;
        }
        return retval;
    }

    @Override
    public Future<Boolean> updateLease(Lease lease) {
        return this.executor.submit(() -> this.updateLeaseSync((InMemoryLease)lease));
    }

    private Boolean updateLeaseSync(InMemoryLease lease) {
        TRACE_LOGGER.info(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "Updating lease"));
        boolean retval = this.renewLeaseSync(lease);
        if (retval) {
            InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(lease.getPartitionId());
            if (leaseInStore != null) {
                if (!this.wrapIsExpired(leaseInStore) && leaseInStore.getOwner().compareTo(this.host.getHostName()) == 0) {
                    leaseInStore.setEpoch(lease.getEpoch());
                    leaseInStore.setToken(lease.getToken());
                } else {
                    TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "updateLease() not updated because we don't own lease"));
                    retval = false;
                }
            } else {
                TRACE_LOGGER.warn(LoggingUtils.withHostAndPartition(this.host.getHostName(), lease.getPartitionId(), "updateLease() can't find lease"));
                retval = false;
            }
        }
        return retval;
    }

    private boolean wrapIsExpired(InMemoryLease lease) {
        boolean retval = false;
        try {
            retval = lease.isExpired();
        }
        catch (Exception exception) {
            // empty catch block
        }
        return retval;
    }

    private static class InMemoryLease
    extends Lease {
        private long expirationTimeMillis = 0L;

        InMemoryLease(String partitionId) {
            super(partitionId);
        }

        InMemoryLease(InMemoryLease source) {
            super(source);
            this.expirationTimeMillis = source.expirationTimeMillis;
        }

        void setExpirationTime(long expireAtMillis) {
            this.expirationTimeMillis = expireAtMillis;
        }

        long getExpirationTime() {
            return this.expirationTimeMillis;
        }

        @Override
        public boolean isExpired() throws Exception {
            boolean hasExpired;
            boolean bl = hasExpired = System.currentTimeMillis() >= this.expirationTimeMillis;
            if (hasExpired) {
                // empty if block
            }
            return hasExpired;
        }
    }

    private static class InMemoryLeaseStore {
        static final InMemoryLeaseStore singleton = new InMemoryLeaseStore();
        private static int leaseDurationInMilliseconds;
        private ConcurrentHashMap<String, InMemoryLease> inMemoryLeasesPrivate = null;

        private InMemoryLeaseStore() {
        }

        synchronized boolean existsMap() {
            return this.inMemoryLeasesPrivate != null;
        }

        synchronized void initializeMap(int leaseDurationInMilliseconds) {
            if (this.inMemoryLeasesPrivate == null) {
                this.inMemoryLeasesPrivate = new ConcurrentHashMap();
            }
            InMemoryLeaseStore.leaseDurationInMilliseconds = leaseDurationInMilliseconds;
        }

        synchronized void deleteMap() {
            this.inMemoryLeasesPrivate = null;
        }

        synchronized InMemoryLease getLease(String partitionId) {
            return this.inMemoryLeasesPrivate.get(partitionId);
        }

        synchronized InMemoryLease atomicAquireUnowned(String partitionId, String newOwner) {
            InMemoryLease leaseInStore = this.getLease(partitionId);
            try {
                if (leaseInStore.isExpired() || leaseInStore.getOwner() == null || leaseInStore.getOwner().isEmpty()) {
                    leaseInStore.setOwner(newOwner);
                    leaseInStore.setExpirationTime(System.currentTimeMillis() + (long)leaseDurationInMilliseconds);
                } else {
                    leaseInStore = null;
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            return leaseInStore;
        }

        synchronized void setOrReplaceLease(InMemoryLease newLease) {
            this.inMemoryLeasesPrivate.put(newLease.getPartitionId(), newLease);
        }

        synchronized void removeLease(InMemoryLease goneLease) {
            this.inMemoryLeasesPrivate.remove(goneLease.getPartitionId());
        }
    }
}

