/**
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 */

package com.microsoft.azure.management.resources.implementation;

import com.google.common.base.Joiner;
import com.microsoft.azure.PagedList;
import com.microsoft.azure.management.resources.ForceDeletionResourceType;
import com.microsoft.azure.management.resources.ResourceGroup;
import com.microsoft.azure.management.resources.ResourceGroups;
import com.microsoft.azure.management.resources.fluentcore.arm.ResourceUtils;
import com.microsoft.azure.management.resources.fluentcore.arm.collection.implementation.CreatableResourcesImpl;
import com.microsoft.azure.management.resources.fluentcore.utils.SdkContext;
import com.microsoft.azure.management.resources.fluentcore.utils.Utils;
import com.microsoft.rest.ServiceFuture;
import com.microsoft.rest.ServiceCallback;
import rx.Completable;
import rx.Observable;

import java.util.Collection;

/**
 * The implementation for ResourceGroups.
 */
final class ResourceGroupsImpl
        extends CreatableResourcesImpl<ResourceGroup, ResourceGroupImpl, ResourceGroupInner>
        implements ResourceGroups {
    private final ResourceGroupsInner client;
    private final ResourceManagementClientImpl serviceClient;

    /**
     * Creates an instance of the implementation.
     *
     * @param serviceClient the inner resource management client
     */
    ResourceGroupsImpl(final ResourceManagementClientImpl serviceClient) {
        this.serviceClient = serviceClient;
        this.client = serviceClient.resourceGroups();
    }

    @Override
    public PagedList<ResourceGroup> list() {
        return wrapList(client.list());
    }

    @Override
    public PagedList<ResourceGroup> listByTag(String tagName, String tagValue) {
        return wrapList(client.list(Utils.createOdataFilterForTags(tagName, tagValue), null));
    }

    @Override
    public Observable<ResourceGroup> listByTagAsync(String tagName, String tagValue) {
        return wrapPageAsync(client.listAsync(Utils.createOdataFilterForTags(tagName, tagValue), null));
    }

    @Override
    public ResourceGroupImpl getByName(String name) {
        return wrapModel(client.get(name));
    }

    @Override
    public void deleteByName(String name) {
        deleteByNameAsync(name).await();
    }

    @Override
    public ServiceFuture<Void> deleteByNameAsync(String name, ServiceCallback<Void> callback) {
        return ServiceFuture.fromResponse(client.deleteWithServiceResponseAsync(name), callback);
    }

    @Override
    public Completable deleteByNameAsync(String name) {
        return client.deleteAsync(name).toCompletable();
    }

    @Override
    public ResourceGroupImpl define(String name) {
        return wrapModel(name);
    }

    @Override
    public boolean checkExistence(String name) {
        return this.contain(name);
    }

    @Override
    public boolean contain(String name) {
        return client.checkExistence(name);
    }

    @Override
    public void deleteByName(String name, Collection<ForceDeletionResourceType> forceDeletionResourceTypes) {
        deleteByNameAsync(name, forceDeletionResourceTypes).await();
    }

    @Override
    public ServiceFuture<Void> deleteByNameAsync(String name, Collection<ForceDeletionResourceType> forceDeletionResourceTypes, ServiceCallback<Void> callback) {
        return ServiceFuture.fromResponse(client.deleteWithServiceResponseAsync(name, forceDeletionTypes(forceDeletionResourceTypes)), callback);
    }

    @Override
    public Completable deleteByNameAsync(String name, Collection<ForceDeletionResourceType> forceDeletionResourceTypes) {
        return client.deleteAsync(name, forceDeletionTypes(forceDeletionResourceTypes)).toCompletable().subscribeOn(SdkContext.getRxScheduler());
    }

    private static String forceDeletionTypes(Collection<ForceDeletionResourceType> forceDeletionResourceTypes) {
        String typesInStr = null;
        if (forceDeletionResourceTypes != null && !forceDeletionResourceTypes.isEmpty()) {
            typesInStr = Joiner.on(",").join(forceDeletionResourceTypes);
        }
        return typesInStr;
    }

    @Override
    protected ResourceGroupImpl wrapModel(String name) {
        return new ResourceGroupImpl(new ResourceGroupInner(), name, serviceClient);
    }

    @Override
    protected ResourceGroupImpl wrapModel(ResourceGroupInner inner) {
        if (inner == null) {
            return null;
        }
        return new ResourceGroupImpl(inner, inner.name(), serviceClient);
    }

    @Override
    public void beginDeleteByName(String id) {
        beginDeleteByNameAsync(id).toBlocking().subscribe();
    }

    @Override
    public ServiceFuture<Void> beginDeleteByNameAsync(String name, ServiceCallback<Void> callback) {
        return ServiceFuture.fromBody(beginDeleteByNameAsync(name), callback);
    }

    @Override
    public Observable<Void> beginDeleteByNameAsync(String name) {
        return client.beginDeleteAsync(name);
    }

    @Override
    public Completable deleteByIdAsync(String id) {
        return deleteByNameAsync(ResourceUtils.nameFromResourceId(id));
    }

    @Override
    public Observable<ResourceGroup> listAsync() {
        return wrapPageAsync(this.client.listAsync());
    }
}
