// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

package com.microsoft.azure.relay;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import org.eclipse.jetty.http.HttpStatus;

public class RelayedHttpListenerResponse {
	private final Map<String, String> headers;
	private final RelayedHttpListenerContext context;
	private boolean disposed;
	private boolean readonly;
	private int statusCode;
	private String statusDescription;
	private HybridHttpConnection.ResponseStream outputStream;

	RelayedHttpListenerResponse(RelayedHttpListenerContext context) {
		this.context = context;
		this.statusCode = HttpStatus.CONTINUE_100;
		this.headers = new ResponseHeaderCollection();
		this.outputStream = null;
	}
	
	public int getStatusCode() {
		return statusCode;
	}

	public void setStatusCode(int statuscode) {
		this.checkClosedOrReadonly();
		if (statuscode < 100 || statuscode > 999) {
			throw RelayLogger.throwingException(new IllegalArgumentException("http status code must be between 100 and 999."), this.context);
		}
		this.statusCode = statuscode;
	}

	public String getStatusDescription() {
		if (this.statusDescription == null) {
			this.statusDescription = (HttpStatus.getMessage(this.statusCode) != null)
					? (HttpStatus.getMessage(this.statusCode))
					: null;
		}
		return statusDescription;
	}

	public void setStatusDescription(String statusDescription) {
		this.checkClosedOrReadonly();
		if (statusDescription == null) {
			throw RelayLogger.throwingException(new IllegalArgumentException("cannot set a null statusDecription."), this.context);
		}

		// Need to verify the status description doesn't contain any control characters
		// except HT.
		for (int i = 0; i < statusDescription.length(); i++) {
			char c = statusDescription.charAt(i);
			if ((c <= 31 && c != '\t') || c >= 127) {
				throw RelayLogger.throwingException(
						new IllegalArgumentException("status description should not contain any control characters."), this.context);
			}
		}
		this.statusDescription = statusDescription;
	}

	public Map<String, String> getHeaders() {
		return headers;
	}

	public HybridHttpConnection.ResponseStream getOutputStream() {
		return outputStream;
	}

	void setOutputStream(HybridHttpConnection.ResponseStream outputStream) {
		this.outputStream = outputStream;
	}

	public RelayedHttpListenerContext getContext() {
		return context;
	}

	/**
	 * Sends the response to the client asynchronously and releases the resources held by this
	 * RelayedHttpListenerResponse instance.
	 */
	public CompletableFuture<Void> closeAsync() {
		if (this.outputStream != null) {
			return this.outputStream.closeAsync()
				.whenComplete(($void, ex) -> this.disposed = true);
		} else {
			this.disposed = true;
			return CompletableFuture.completedFuture(null);
		}
	}
	
	/**
	 * Sends the response to the client and releases the resources held by this
	 * RelayedHttpListenerResponse instance.
	 */
	public void close() {
		this.closeAsync().join();
	}

	void setReadonly() {
		this.readonly = true;
	}
	
	private void checkClosedOrReadonly() {
		Exception ex = null;
		if (this.disposed) {
			ex = new IOException("The HTTP status code, status description, and headers cannot be modified after the output stream has already been closed.");
		} else if (this.readonly) {
			ex = new IllegalStateException("The HTTP status code, status description, and headers cannot be modified after writing to the output.");
		}
		
		if (ex != null) {
			throw RelayLogger.throwingException(ex, this.context);
		}
	}
	
	public class ResponseHeaderCollection extends HashMap<String, String> {
		private static final long serialVersionUID = -4334547975481290294L; // autogenerated

		@Override
		public String put(String key, String value) {
			checkClosedOrReadonly();
			return super.put(key, value);
		}

		@Override
		public String remove(Object key) {
			checkClosedOrReadonly();
			return super.remove(key);
		}

		@Override
		public void putAll(Map<? extends String, ? extends String> m) {
			checkClosedOrReadonly();
			super.putAll(m);
		}

		@Override
		public void clear() {
			checkClosedOrReadonly();
			super.clear();
		}
	}
}
