/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.relay;

import com.microsoft.azure.relay.AutoShutdownScheduledExecutor;
import com.microsoft.azure.relay.CompletableFutureUtil;
import com.microsoft.azure.relay.InputQueue;
import com.microsoft.azure.relay.RelayLogger;
import com.microsoft.azure.relay.RelayTraceSource;
import com.microsoft.azure.relay.TimeoutHelper;
import com.microsoft.azure.relay.TrackingContext;
import com.microsoft.azure.relay.WriteMode;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.LinkedList;
import java.util.concurrent.CompletableFuture;
import javax.websocket.ClientEndpointConfig;
import javax.websocket.CloseReason;
import javax.websocket.ContainerProvider;
import javax.websocket.DeploymentException;
import javax.websocket.Endpoint;
import javax.websocket.EndpointConfig;
import javax.websocket.MessageHandler;
import javax.websocket.OnClose;
import javax.websocket.OnError;
import javax.websocket.OnOpen;
import javax.websocket.RemoteEndpoint;
import javax.websocket.Session;
import javax.websocket.WebSocketContainer;
import org.eclipse.jetty.io.RuntimeIOException;
import org.eclipse.jetty.util.component.LifeCycle;
import org.eclipse.jetty.websocket.api.UpgradeException;

class ClientWebSocket
extends Endpoint
implements RelayTraceSource {
    private final AutoShutdownScheduledExecutor executor;
    private final WebSocketContainer container = ContainerProvider.getWebSocketContainer();
    private final TrackingContext trackingContext;
    private Session session;
    private int maxMessageBufferSize = 65536;
    private CloseReason closeReason;
    private InputQueue<MessageFragment> fragmentQueue;
    private InputQueue<String> textQueue;
    private CompletableFuture<Void> closeTask;
    private String cachedString;

    public ClientWebSocket(TrackingContext trackingContext, AutoShutdownScheduledExecutor executor) {
        this.executor = executor;
        this.textQueue = new InputQueue(this.executor);
        this.fragmentQueue = new InputQueue(this.executor);
        this.closeReason = null;
        this.trackingContext = trackingContext;
    }

    @Override
    public TrackingContext getTrackingContext() {
        return this.trackingContext;
    }

    @Override
    public String toString() {
        if (this.cachedString == null) {
            this.cachedString = this.getClass().getSimpleName() + "(" + this.trackingContext + ")";
        }
        return this.cachedString;
    }

    CloseReason getCloseReason() {
        return this.closeReason;
    }

    int getMaxMessageBufferSize() {
        return this.maxMessageBufferSize;
    }

    void setMaxMessageBufferSize(int maxMessageBufferSize) {
        if (maxMessageBufferSize <= 0) {
            throw new IllegalArgumentException("MaxBufferSize of the web socket must be a positive value.");
        }
        this.maxMessageBufferSize = maxMessageBufferSize;
        this.container.setDefaultMaxTextMessageBufferSize(this.maxMessageBufferSize);
    }

    public CompletableFuture<Void> connectAsync(URI uri) {
        return this.connectAsync(uri, null, null);
    }

    public CompletableFuture<Void> connectAsync(URI uri, Duration timeout) {
        return this.connectAsync(uri, timeout, null);
    }

    public CompletableFuture<Void> connectAsync(URI uri, Duration timeout, ClientEndpointConfig config) {
        if (this.isOpen()) {
            return CompletableFutureUtil.fromException((Throwable)new RuntimeIOException("This connection is already connected."));
        }
        this.container.setDefaultMaxTextMessageBufferSize(this.maxMessageBufferSize);
        return CompletableFutureUtil.timedRunAsync(timeout, () -> {
            RelayLogger.logEvent("connecting", this, new String[0]);
            try {
                if (config != null) {
                    this.container.connectToServer((Endpoint)this, config, uri);
                } else {
                    this.container.connectToServer((Object)this, uri);
                }
            }
            catch (IOException | DeploymentException e) {
                if (e.getCause() instanceof UpgradeException) {
                    throw RelayLogger.throwingException(e.getCause(), this);
                }
                throw RelayLogger.throwingException(e, this);
            }
            if (this.session == null || !this.session.isOpen()) {
                throw RelayLogger.throwingException((Throwable)new RuntimeIOException("connection to the server failed."), this);
            }
        }, this.executor).whenComplete(($void, ex) -> {
            if (ex != null) {
                this.dispose();
            }
        });
    }

    boolean isOpen() {
        return this.session != null && this.session.isOpen();
    }

    CompletableFuture<String> readTextAsync() {
        return this.textQueue.dequeueAsync().thenApply(text -> {
            if (text != null) {
                RelayLogger.logEvent("receivedText", this, String.valueOf(text.length()));
            }
            return text;
        });
    }

    public CompletableFuture<ByteBuffer> readBinaryAsync() {
        return this.readBinaryAsync(null);
    }

    public CompletableFuture<ByteBuffer> readBinaryAsync(Duration timeout) {
        BinaryMessageReader messageReader = new BinaryMessageReader(timeout);
        return messageReader.readAsync();
    }

    public CompletableFuture<Void> writeAsync(Object data) {
        return this.writeAsync(data, null);
    }

    public CompletableFuture<Void> writeAsync(Object data, Duration timeout) {
        return this.writeAsync(data, timeout, true, WriteMode.BINARY);
    }

    CompletableFuture<Void> writeAsync(Object data, Duration timeout, boolean isEnd, WriteMode mode) {
        if (this.isOpen()) {
            if (data == null) {
                return CompletableFuture.completedFuture(null);
            }
            RemoteEndpoint.Basic remote = this.session.getBasicRemote();
            RelayLogger.logEvent("writingBytes", this, mode.toString());
            return CompletableFutureUtil.timedRunAsync(timeout, () -> {
                try {
                    if (mode.equals((Object)WriteMode.TEXT)) {
                        String text = data.toString();
                        remote.sendText(text, isEnd);
                        RelayLogger.logEvent("writingBytesFinished", this, String.valueOf(text.length()));
                    } else {
                        byte[] bytes;
                        if (data instanceof byte[]) {
                            bytes = (byte[])((byte[])data).clone();
                        } else if (data instanceof ByteBuffer) {
                            ByteBuffer buffer = (ByteBuffer)data;
                            bytes = new byte[buffer.remaining()];
                            buffer.get(bytes);
                        } else {
                            throw new IllegalArgumentException("The data to be sent should be ByteBuffer or byte[], but received " + data.getClass().getSimpleName());
                        }
                        int bytesToSend = bytes.length;
                        remote.sendBinary(ByteBuffer.wrap(bytes), isEnd);
                        RelayLogger.logEvent("writingBytesFinished", this, String.valueOf(bytesToSend));
                    }
                }
                catch (Exception e) {
                    throw RelayLogger.throwingException(e, this);
                }
            }, this.executor);
        }
        return CompletableFutureUtil.fromException((Throwable)new RuntimeIOException("cannot send because the session is not connected."));
    }

    public CompletableFuture<Void> closeAsync() {
        return this.closeAsync(null);
    }

    public CompletableFuture<Void> closeAsync(CloseReason reason) {
        RelayLogger.logEvent("clientWebSocketClosing", this, reason != null ? reason.getReasonPhrase() : "NONE");
        if (this.session == null || !this.session.isOpen()) {
            return this.closeTask;
        }
        try {
            if (reason != null) {
                this.session.close(reason);
            } else {
                this.session.close();
            }
        }
        catch (Throwable e) {
            this.closeTask.completeExceptionally(e);
        }
        return this.closeTask;
    }

    void dispose() {
        try {
            ((LifeCycle)this.container).stop();
        }
        catch (Exception e) {
            RelayLogger.handledExceptionAsWarning(e, this);
        }
    }

    @OnOpen
    public void onOpen(Session session, EndpointConfig config) {
        RelayLogger.logEvent("connected", this, new String[0]);
        this.closeReason = null;
        this.session = session;
        session.setMaxBinaryMessageBufferSize(this.maxMessageBufferSize);
        session.setMaxTextMessageBufferSize(this.maxMessageBufferSize);
        this.closeTask = new CompletableFuture();
        session.addMessageHandler((MessageHandler)new MessageHandler.Whole<String>(){

            public void onMessage(String text) {
                ClientWebSocket.this.textQueue.enqueueAndDispatch(text);
            }
        });
        session.addMessageHandler((MessageHandler)new MessageHandler.Partial<byte[]>(){

            public void onMessage(byte[] inputBytes, boolean isEnd) {
                ClientWebSocket.this.fragmentQueue.enqueueAndDispatch(new MessageFragment(inputBytes, isEnd));
            }
        });
    }

    @OnClose
    public void onClose(Session session, CloseReason reason) {
        CompletableFuture.runAsync(() -> this.dispose(), this.executor);
        this.closeReason = reason;
        RelayLogger.logEvent("clientWebSocketClosed", this, reason.getReasonPhrase());
        this.textQueue.shutdown();
        this.fragmentQueue.shutdown();
        this.closeTask.complete(null);
    }

    @OnError
    public void onError(Throwable cause) {
        if (!this.isOpen()) {
            CompletableFuture.runAsync(() -> this.dispose(), this.executor);
        }
        RelayLogger.throwingException(cause, this);
    }

    private final class BinaryMessageReader {
        private final TimeoutHelper timeoutHelper;
        private final LinkedList<byte[]> fragments;
        private int messageSize;

        BinaryMessageReader(Duration timeout) {
            this.timeoutHelper = new TimeoutHelper(timeout);
            this.fragments = new LinkedList();
        }

        public CompletableFuture<ByteBuffer> readAsync() {
            return this.readFragmentsAsync().thenApply(voidResult -> {
                byte[] message = new byte[this.messageSize];
                int offset = 0;
                for (byte[] bytes : this.fragments) {
                    System.arraycopy(bytes, 0, message, offset, bytes.length);
                    offset += bytes.length;
                }
                RelayLogger.logEvent("receivedBytes", this, Integer.toString(message.length));
                return ByteBuffer.wrap(message);
            });
        }

        private CompletableFuture<Void> readFragmentsAsync() {
            return ClientWebSocket.this.fragmentQueue.dequeueAsync(this.timeoutHelper.remainingTime()).thenCompose(fragment -> {
                if (fragment == null) {
                    return CompletableFuture.completedFuture(null);
                }
                this.messageSize += fragment.getBytes().length;
                this.fragments.add(fragment.getBytes());
                if (!fragment.isEnd()) {
                    return this.readFragmentsAsync();
                }
                return CompletableFuture.completedFuture(null);
            });
        }
    }

    private static class MessageFragment {
        private final byte[] bytes;
        private final boolean ended;

        MessageFragment(byte[] bytes, boolean ended) {
            this.bytes = bytes;
            this.ended = ended;
        }

        byte[] getBytes() {
            return this.bytes;
        }

        boolean isEnd() {
            return this.ended;
        }
    }
}

