/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.relay;

import com.microsoft.azure.relay.CompletableFutureUtil;
import com.microsoft.azure.relay.SecurityToken;
import com.microsoft.azure.relay.StringUtil;
import com.microsoft.azure.relay.TokenProvider;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

public class SharedAccessSignatureTokenProvider
extends TokenProvider {
    private static final String UTF8_ENCODING_NAME = StandardCharsets.UTF_8.name();
    private final byte[] encodedSharedAccessKey;
    private final String keyName;
    private final String sharedAccessSignature;

    public SharedAccessSignatureTokenProvider(String sharedAccessSignature) {
        SharedAccessSignatureToken.validate(sharedAccessSignature);
        this.encodedSharedAccessKey = null;
        this.keyName = null;
        this.sharedAccessSignature = sharedAccessSignature;
    }

    public SharedAccessSignatureTokenProvider(String keyName, String sharedAccessKey) {
        this(keyName, sharedAccessKey, null);
    }

    public SharedAccessSignatureTokenProvider(String keyName, String sharedAccessKey, Charset charset) {
        this.sharedAccessSignature = null;
        if (StringUtil.isNullOrEmpty(keyName) || StringUtil.isNullOrEmpty(sharedAccessKey)) {
            throw new IllegalArgumentException("keyName or key cannot be empty.");
        }
        if (keyName.length() > 256) {
            throw new IllegalArgumentException("length of keyName is " + keyName.length() + ", which exceeded the maximum of " + 256);
        }
        if (sharedAccessKey.length() > 256) {
            throw new IllegalArgumentException("length of keyName is " + sharedAccessKey.length() + ", which exceeded the maximum of " + 256);
        }
        this.keyName = keyName;
        Charset encodingCharset = charset == null ? StringUtil.UTF8 : charset;
        this.encodedSharedAccessKey = sharedAccessKey.getBytes(encodingCharset);
    }

    @Override
    protected CompletableFuture<SecurityToken> onGetTokenAsync(String resource, Duration validFor) {
        try {
            String tokenString = this.buildSignature(resource, validFor);
            SharedAccessSignatureToken securityToken = new SharedAccessSignatureToken(tokenString);
            return CompletableFuture.completedFuture(securityToken);
        }
        catch (UnsupportedEncodingException | InvalidKeyException | NoSuchAlgorithmException e) {
            return CompletableFutureUtil.fromException(e);
        }
    }

    protected String buildSignature(String resource, Duration validFor) throws InvalidKeyException, UnsupportedEncodingException, NoSuchAlgorithmException {
        if (StringUtil.isNullOrWhiteSpace(this.sharedAccessSignature)) {
            return SharedAccessSignatureBuilder.buildSignature(this.keyName, this.encodedSharedAccessKey, resource, validFor);
        }
        return this.sharedAccessSignature;
    }

    static class SharedAccessSignatureToken
    extends SecurityToken {
        public static final int MAX_KEYNAME_LENGTH = 256;
        public static final int MAX_KEY_LENGTH = 256;
        public static final String SHARED_ACCESS_SIGNATURE = "SharedAccessSignature";
        public static final String SIGNED_RESOURCE = "sr";
        public static final String SIGNATURE = "sig";
        public static final String SIGNATURE_KEYNAME = "skn";
        public static final String SIGNED_EXPIRY = "se";
        public static final String SIGNED_RESOURCE_FULL_FIELD_NAME = "SharedAccessSignature sr";
        public static final String SAS_KEY_VALUE_SEPARATOR = "=";
        public static final String SAS_PAIR_SEPARATOR = "&";

        public SharedAccessSignatureToken(String tokenString) {
            super(tokenString);
            this.getExpirationDateAndAudienceFromToken(tokenString);
        }

        private void getExpirationDateAndAudienceFromToken(String tokenString) {
            HashMap<String, String> decodedToken = SharedAccessSignatureToken.getDecodedTokenMap(tokenString, StandardCharsets.UTF_8.name(), StandardCharsets.UTF_8.name(), SAS_KEY_VALUE_SEPARATOR, SAS_PAIR_SEPARATOR);
            String expiresOn = decodedToken.get(SIGNED_EXPIRY);
            if (StringUtil.isNullOrEmpty(expiresOn)) {
                throw new IllegalArgumentException("tokenString missing expiresOn field");
            }
            this.setExiresAtUtc(Instant.ofEpochSecond(Long.parseLong(expiresOn)));
            String audience = decodedToken.get(SIGNED_RESOURCE_FULL_FIELD_NAME);
            if (StringUtil.isNullOrEmpty(audience)) {
                throw new IllegalArgumentException("tokenstring missing audience field");
            }
            this.setAudience(audience);
        }

        private static HashMap<String, String> getDecodedTokenMap(String tokenString, String keyEncodingScheme, String valueEncodingScheme, String keyValueSeparator, String pairSeparator) {
            String[] valueEncodedPairs;
            HashMap<String, String> map = new HashMap<String, String>();
            for (String valueEncodedPair : valueEncodedPairs = tokenString.split(pairSeparator)) {
                String[] pair = valueEncodedPair.split(keyValueSeparator);
                if (pair.length != 2) {
                    throw new IllegalArgumentException("invalid encoding of tokenString.");
                }
                try {
                    map.put(URLDecoder.decode(pair[0], keyEncodingScheme), URLDecoder.decode(pair[1], valueEncodingScheme));
                }
                catch (UnsupportedEncodingException e) {
                    throw new RuntimeException(keyEncodingScheme + (keyEncodingScheme.equals(valueEncodingScheme) ? "" : " or " + valueEncodingScheme) + " decoding is not supported in the java runtime.");
                }
            }
            return map;
        }

        static void validate(String sharedAccessSignature) {
            if (StringUtil.isNullOrEmpty(sharedAccessSignature)) {
                throw new IllegalArgumentException("sharedAccessSignature cannot be null or empty");
            }
            Map<String, String> parsedFields = SharedAccessSignatureToken.extractFieldValues(sharedAccessSignature);
            for (String field : new String[]{SIGNATURE, SIGNED_EXPIRY, SIGNATURE_KEYNAME, SIGNED_RESOURCE}) {
                if (parsedFields.get(field) != null) continue;
                throw new IllegalArgumentException(field.toString() + " has no corresponding value.");
            }
        }

        static Map<String, String> extractFieldValues(String sharedAccessSignature) {
            String[] tokenFields;
            String[] tokenLines = sharedAccessSignature.split("\\s+");
            if (!tokenLines[0].trim().equalsIgnoreCase(SHARED_ACCESS_SIGNATURE) || tokenLines.length != 2) {
                throw new IllegalArgumentException("invalid sharedAccessSignture.");
            }
            HashMap<String, String> parsedFields = new HashMap<String, String>();
            for (String tokenField : tokenFields = tokenLines[1].trim().split(SAS_PAIR_SEPARATOR)) {
                if (StringUtil.isNullOrEmpty(tokenField)) continue;
                String[] fieldParts = tokenField.split(SAS_KEY_VALUE_SEPARATOR);
                String key = fieldParts[0].toLowerCase();
                try {
                    parsedFields.put(key, key.equalsIgnoreCase(SIGNED_RESOURCE) ? fieldParts[1] : URLDecoder.decode(fieldParts[1], UTF8_ENCODING_NAME));
                }
                catch (UnsupportedEncodingException e) {
                    throw new RuntimeException("UTF-8 decoding is not supported in the java runtime.");
                }
            }
            return parsedFields;
        }
    }

    static class SharedAccessSignatureBuilder {
        static final String HMAC_ALGORITHM = "HMACSHA256";

        SharedAccessSignatureBuilder() {
        }

        public static String buildSignature(String keyName, byte[] encodedSharedAccessKey, String resource, Duration timeToLive) throws UnsupportedEncodingException, InvalidKeyException, NoSuchAlgorithmException {
            String expiresOn = SharedAccessSignatureBuilder.buildExpiresOn(timeToLive);
            String audienceUri = URLEncoder.encode(resource, UTF8_ENCODING_NAME);
            CharSequence[] fields = new String[]{audienceUri, expiresOn};
            String signature = SharedAccessSignatureBuilder.sign(String.join((CharSequence)"\n", fields), encodedSharedAccessKey);
            return String.format(Locale.ROOT, "%s %s=%s&%s=%s&%s=%s&%s=%s", "SharedAccessSignature", "sr", audienceUri, "sig", URLEncoder.encode(signature, UTF8_ENCODING_NAME), "se", URLEncoder.encode(expiresOn, UTF8_ENCODING_NAME), "skn", URLEncoder.encode(keyName, UTF8_ENCODING_NAME));
        }

        static String buildExpiresOn(Duration timeToLive) {
            long timeToLiveInSeconds = timeToLive.getSeconds();
            if (timeToLiveInSeconds < 0L) {
                throw new IllegalArgumentException("timeToLive should be a positive value");
            }
            long expireOnInSeconds = Instant.now().getEpochSecond() + timeToLiveInSeconds;
            return String.valueOf(expireOnInSeconds);
        }

        static String sign(String requestString, byte[] encodedSharedAccessKey) throws InvalidKeyException, NoSuchAlgorithmException {
            Mac hmac = Mac.getInstance(HMAC_ALGORITHM);
            SecretKeySpec secretKey = new SecretKeySpec(encodedSharedAccessKey, HMAC_ALGORITHM);
            hmac.init(secretKey);
            byte[] signatureBytes = hmac.doFinal(requestString.getBytes(StringUtil.UTF8));
            return Base64.getEncoder().encodeToString(signatureBytes);
        }
    }
}

