/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.servicebus.jms;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketOption;
import java.net.StandardSocketOptions;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousServerSocketChannel;
import java.nio.channels.AsynchronousSocketChannel;
import java.nio.channels.Channel;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.CompletionHandler;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestProxy
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(TestProxy.class);
    private static final int TIMEOUT_IN_S = 2;
    private AsynchronousServerSocketChannel serverSocketChannel;
    private AtomicInteger connectCount = new AtomicInteger();
    private ProxyReadHandler readHandler = new ProxyReadHandler();
    private ProxyWriteHandler writeHandler = new ProxyWriteHandler();
    private int port;
    private final ProxyType type;

    public TestProxy(ProxyType type) throws IOException {
        Objects.requireNonNull(type, "Proxy type must be given");
        this.type = type;
    }

    public int getPort() {
        return this.port;
    }

    public int getSuccessCount() {
        return this.connectCount.get();
    }

    public void start() throws IOException {
        this.serverSocketChannel = AsynchronousServerSocketChannel.open();
        this.serverSocketChannel.bind(new InetSocketAddress(0));
        this.port = ((InetSocketAddress)this.serverSocketChannel.getLocalAddress()).getPort();
        LOG.info("Bound listen socket to port {}, waiting for clients...", (Object)this.port);
        this.serverSocketChannel.accept(null, new ServerConnectionHandler());
    }

    @Override
    public void close() {
        LOG.info("stopping proxy server");
        if (this.serverSocketChannel != null) {
            try {
                LOG.info("Terminating server socket");
                this.serverSocketChannel.close();
            }
            catch (Exception e) {
                LOG.error("Cannot close server socket ", (Throwable)e);
            }
        }
    }

    private boolean processHandshakeMessages(ProxyConnectionState attachment) {
        if (attachment.handshakePhase == HandshakePhase.INITIAL) {
            byte first = attachment.buffer.get(0);
            attachment.handshakePhase = first == 5 ? HandshakePhase.SOCKS5_1 : HandshakePhase.HTTP;
        }
        if (!this.assertExpectedHandshakeType(attachment.handshakePhase)) {
            LOG.error("Unexpected handshake phase '" + (Object)((Object)attachment.handshakePhase) + "' for proxy of type: " + (Object)((Object)this.type));
            return false;
        }
        switch (attachment.handshakePhase) {
            case SOCKS5_1: {
                return this.processSocks5Handshake1(attachment);
            }
            case SOCKS5_2: {
                return this.processSocks5Handshake2(attachment);
            }
            case HTTP: {
                return this.processHttpHandshake(attachment);
            }
        }
        LOG.error("wrong handshake phase");
        return false;
    }

    private boolean assertExpectedHandshakeType(HandshakePhase handshakePhase) {
        switch (handshakePhase) {
            case SOCKS5_1: 
            case SOCKS5_2: {
                return this.type == ProxyType.SOCKS5;
            }
            case HTTP: {
                return this.type == ProxyType.HTTP;
            }
        }
        LOG.error("Unknown handshake phase type:" + (Object)((Object)handshakePhase));
        return false;
    }

    private boolean processHttpHandshake(ProxyConnectionState attachment) {
        String line;
        String requestString = StandardCharsets.ISO_8859_1.decode(attachment.buffer).toString();
        LOG.debug("Request received: {}", (Object)requestString);
        String requestType = requestString.substring(0, requestString.indexOf(32));
        String hostandport = requestString.substring(requestType.length() + 1);
        hostandport = hostandport.substring(0, hostandport.indexOf(32));
        String hostname = hostandport.substring(0, hostandport.indexOf(":"));
        int port = Integer.parseInt(hostandport.substring(hostname.length() + 1));
        if (requestType.equals("CONNECT")) {
            LOG.info("CONNECT to {}:{}", (Object)hostname, (Object)port);
            if (this.connectToServer(hostname, port, attachment)) {
                attachment.handshakePhase = HandshakePhase.CONNECTED;
                line = "HTTP/1.1 200 Connection established\r\n\r\n";
            } else {
                line = "HTTP/1.1 504 Gateway Timeout\r\n\r\n";
            }
        } else {
            LOG.error("unsupported request type {}", (Object)requestType);
            line = "HTTP/1.1 502 Bad Gateway\r\n\r\n";
        }
        attachment.buffer.clear();
        attachment.buffer.put(StandardCharsets.ISO_8859_1.encode(line));
        attachment.buffer.flip();
        return true;
    }

    private boolean processSocks5Handshake1(ProxyConnectionState attachment) {
        byte version = attachment.buffer.get();
        if (version != 5) {
            LOG.error("SOCKS Version {} not supported", (Object)version);
            this.closeChannel(attachment.readChannel);
            return false;
        }
        attachment.buffer.clear();
        attachment.buffer.put(version);
        attachment.buffer.put((byte)0);
        attachment.buffer.flip();
        LOG.info("SOCKS5 connection initialized, no authentication required");
        attachment.handshakePhase = HandshakePhase.SOCKS5_2;
        return true;
    }

    private boolean processSocks5Handshake2(ProxyConnectionState attachment) {
        byte version = attachment.buffer.get();
        if (version != 5) {
            LOG.error("SOCKS Version {} not supported", (Object)version);
            this.closeChannel(attachment.readChannel);
            return false;
        }
        byte command = attachment.buffer.get();
        if (command != 1) {
            LOG.error("CMD {} not supported", (Object)command);
            this.closeChannel(attachment.readChannel);
            return false;
        }
        attachment.buffer.get();
        byte addressType = attachment.buffer.get();
        if (addressType != 3) {
            LOG.error("Address Type {} not supported", (Object)addressType);
            this.closeChannel(attachment.readChannel);
            return false;
        }
        int size = attachment.buffer.get() & 0xFF;
        byte[] hostBytes = new byte[size];
        attachment.buffer.get(hostBytes);
        String hostname = new String(hostBytes, StandardCharsets.UTF_8);
        int port = attachment.buffer.getShort() & 0xFFFF;
        LOG.info("Create SOCKS5 connection to {}:{}", (Object)hostname, (Object)port);
        if (!this.connectToServer(hostname, port, attachment)) {
            return false;
        }
        attachment.buffer.rewind();
        attachment.buffer.put(1, (byte)0);
        attachment.handshakePhase = HandshakePhase.CONNECTED;
        return true;
    }

    private boolean isInHandshake(ProxyConnectionState attachment) {
        return attachment.handshakePhase != HandshakePhase.CONNECTED;
    }

    private boolean connectToServer(String hostname, int port, ProxyConnectionState attachment) {
        try {
            AsynchronousSocketChannel serverChannel = AsynchronousSocketChannel.open();
            try {
                serverChannel.setOption((SocketOption)StandardSocketOptions.TCP_NODELAY, (Object)true);
            }
            catch (IOException e) {
                LOG.error("Failed to set TCP_NODELAY before connect, closing channel", (Throwable)e);
                this.closeChannel(serverChannel);
                return false;
            }
            InetSocketAddress serverAddr = new InetSocketAddress(hostname, port);
            Future<Void> connectResult = serverChannel.connect(serverAddr);
            connectResult.get(2L, TimeUnit.SECONDS);
            attachment.writeChannel = serverChannel;
            int connectionNumber = this.connectCount.incrementAndGet();
            LOG.info("Connection {} to {}:{} established", new Object[]{connectionNumber, hostname, port});
            ProxyConnectionState serverState = new ProxyConnectionState();
            serverState.readChannel = attachment.writeChannel;
            serverState.writeChannel = attachment.readChannel;
            serverState.buffer = ByteBuffer.allocate(4096);
            serverState.handshakePhase = HandshakePhase.CONNECTED;
            serverState.readChannel.read(serverState.buffer, 2L, TimeUnit.SECONDS, serverState, this.readHandler);
            return true;
        }
        catch (IOException | InterruptedException | ExecutionException | TimeoutException e) {
            LOG.error("connection failed ", (Throwable)e);
            this.closeChannel(attachment.readChannel);
            return false;
        }
    }

    private void closeChannel(Channel channel) {
        if (channel != null && channel.isOpen()) {
            try {
                channel.close();
            }
            catch (IOException e) {
                LOG.error("cannot close", (Throwable)e);
            }
        }
    }

    private void shutdownOutput(AsynchronousSocketChannel channel) {
        if (channel != null && channel.isOpen()) {
            try {
                LOG.info("shutdown output for ({})", (Object)channel);
                channel.shutdownOutput();
            }
            catch (IOException e) {
                LOG.error("cannot shutdown output to ({})", (Object)channel, (Object)e);
            }
        }
    }

    private class ProxyWriteHandler
    implements CompletionHandler<Integer, ProxyConnectionState> {
        private ProxyWriteHandler() {
        }

        @Override
        public void completed(Integer result, ProxyConnectionState attachment) {
            if (result == -1) {
                LOG.info("write connection closed");
                TestProxy.this.closeChannel(attachment.writeChannel);
                TestProxy.this.closeChannel(attachment.readChannel);
                return;
            }
            LOG.debug("wrote {} bytes", (Object)result);
            attachment.buffer.clear();
            if (attachment.readChannel.isOpen()) {
                attachment.readChannel.read(attachment.buffer, attachment, TestProxy.this.readHandler);
            } else {
                TestProxy.this.closeChannel(attachment.writeChannel);
            }
        }

        @Override
        public void failed(Throwable e, ProxyConnectionState attachment) {
            if (!(e instanceof ClosedChannelException)) {
                LOG.info("write failed", e);
            }
            TestProxy.this.closeChannel(attachment.writeChannel);
            TestProxy.this.closeChannel(attachment.readChannel);
        }
    }

    private class ProxyReadHandler
    implements CompletionHandler<Integer, ProxyConnectionState> {
        private ProxyReadHandler() {
        }

        @Override
        public void completed(Integer result, ProxyConnectionState attachment) {
            if (result == -1) {
                LOG.info("read connection reached end of file ({})", (Object)attachment.readChannel);
                TestProxy.this.closeChannel(attachment.readChannel);
                TestProxy.this.shutdownOutput(attachment.writeChannel);
                return;
            }
            LOG.info("read {} bytes (from {})", (Object)result, (Object)attachment.readChannel);
            attachment.buffer.flip();
            if (TestProxy.this.isInHandshake(attachment)) {
                if (TestProxy.this.processHandshakeMessages(attachment)) {
                    attachment.readChannel.write(attachment.buffer, 2L, TimeUnit.SECONDS, attachment, TestProxy.this.writeHandler);
                }
            } else {
                if (attachment.writeChannel == null) {
                    LOG.error("Invalid");
                    TestProxy.this.closeChannel(attachment.readChannel);
                    return;
                }
                if (attachment.writeChannel.isOpen()) {
                    attachment.writeChannel.write(attachment.buffer, 2L, TimeUnit.SECONDS, attachment, TestProxy.this.writeHandler);
                } else {
                    TestProxy.this.closeChannel(attachment.readChannel);
                }
            }
        }

        @Override
        public void failed(Throwable e, ProxyConnectionState attachment) {
            if (!(e instanceof ClosedChannelException)) {
                LOG.info("read failed", e);
            }
            TestProxy.this.closeChannel(attachment.writeChannel);
            TestProxy.this.closeChannel(attachment.readChannel);
        }
    }

    private class ServerConnectionHandler
    implements CompletionHandler<AsynchronousSocketChannel, Object> {
        private ServerConnectionHandler() {
        }

        @Override
        public void completed(AsynchronousSocketChannel clientChannel, Object attachment) {
            TestProxy.this.serverSocketChannel.accept(attachment, this);
            ProxyConnectionState clientState = new ProxyConnectionState();
            clientState.readChannel = clientChannel;
            clientState.buffer = ByteBuffer.allocate(4096);
            clientState.handshakePhase = HandshakePhase.INITIAL;
            try {
                clientChannel.setOption((SocketOption)StandardSocketOptions.TCP_NODELAY, (Object)true);
            }
            catch (IOException e) {
                LOG.error("Failed to set TCP_NODELAY after accept, closing channel", (Throwable)e);
                TestProxy.this.closeChannel(clientChannel);
                return;
            }
            clientChannel.read(clientState.buffer, 2L, TimeUnit.SECONDS, clientState, TestProxy.this.readHandler);
        }

        @Override
        public void failed(Throwable e, Object attachment) {
            if (!(e instanceof ClosedChannelException)) {
                LOG.error("failed to accept connection ", e);
            }
            TestProxy.this.closeChannel(TestProxy.this.serverSocketChannel);
        }
    }

    private static enum HandshakePhase {
        INITIAL,
        HTTP,
        SOCKS5_1,
        SOCKS5_2,
        CONNECTED;

    }

    private static class ProxyConnectionState {
        AsynchronousSocketChannel readChannel;
        AsynchronousSocketChannel writeChannel;
        ByteBuffer buffer;
        HandshakePhase handshakePhase;

        private ProxyConnectionState() {
        }
    }

    public static enum ProxyType {
        SOCKS5,
        HTTP;

    }
}

