/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.toolkit.lib.database;

import com.google.common.base.Preconditions;
import com.microsoft.azure.toolkit.lib.common.exception.AzureToolkitRuntimeException;
import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Objects;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;

public abstract class JdbcUrl {
    private static final int MYSQL_DEFAULT_PORT = 3306;
    private static final int POSTGRE_SQL_DEFAULT_PORT = 5432;
    private static final int SQL_SERVER_DEFAULT_PORT = 1433;
    @Nonnull
    protected final URIBuilder uri;
    private String username;
    private String password;

    private JdbcUrl(@Nonnull String url) {
        Preconditions.checkArgument((boolean)StringUtils.startsWith((CharSequence)url, (CharSequence)"jdbc:"), (Object)"invalid jdbc url.");
        try {
            this.uri = new URIBuilder(url.substring(5));
        }
        catch (URISyntaxException e) {
            throw new AzureToolkitRuntimeException("invalid jdbc url: %s", url);
        }
    }

    @Nonnull
    public static JdbcUrl from(@Nonnull String connectionString) {
        if (StringUtils.startsWith((CharSequence)connectionString, (CharSequence)"jdbc:mysql:")) {
            return new MySQLJdbcUrl(connectionString);
        }
        if (StringUtils.startsWith((CharSequence)connectionString, (CharSequence)"jdbc:sqlserver:")) {
            return new SQLServerJdbcUrl(connectionString);
        }
        if (StringUtils.startsWith((CharSequence)connectionString, (CharSequence)"jdbc:postgresql:")) {
            return new PostgreSQLJdbcUrl(connectionString);
        }
        throw new AzureToolkitRuntimeException("Unsupported jdbc url: %s", connectionString);
    }

    @Nonnull
    public static JdbcUrl mysql(@Nonnull String serverHost, @Nonnull String database) {
        return new MySQLJdbcUrl(String.format("jdbc:mysql://%s:%s/%s?serverTimezone=UTC&useSSL=true&requireSSL=false", JdbcUrl.encode(serverHost), 3306, JdbcUrl.encode(database)));
    }

    @Nonnull
    public static JdbcUrl mysql(@Nonnull String serverHost) {
        return new MySQLJdbcUrl(String.format("jdbc:mysql://%s:%s?serverTimezone=UTC&useSSL=true&requireSSL=false", JdbcUrl.encode(serverHost), 3306));
    }

    @Nonnull
    public static JdbcUrl postgre(@Nonnull String serverHost, @Nonnull String database) {
        return new PostgreSQLJdbcUrl(String.format("jdbc:postgresql://%s:%s/%s?ssl=true&sslmode=require", JdbcUrl.encode(serverHost), 5432, JdbcUrl.encode(database)));
    }

    @Nonnull
    public static JdbcUrl sqlserver(@Nonnull String serverHost, @Nonnull String database) {
        return new SQLServerJdbcUrl(String.format("jdbc:sqlserver://%s:%s;encrypt=true;trustServerCertificate=false;loginTimeout=30;database=%s;", JdbcUrl.encode(serverHost), 1433, JdbcUrl.encode(database)));
    }

    @Nonnull
    public static JdbcUrl sqlserver(@Nonnull String serverHost) {
        return new SQLServerJdbcUrl(String.format("jdbc:sqlserver://%s:%s;encrypt=true;trustServerCertificate=false;loginTimeout=30;", JdbcUrl.encode(serverHost), 1433));
    }

    abstract int getDefaultPort();

    @Nonnull
    public abstract String getDefaultDriverClass();

    public int getPort() {
        if (this.uri.getPort() >= 0) {
            return this.uri.getPort();
        }
        return this.getDefaultPort();
    }

    public String getServerHost() {
        return JdbcUrl.decode(this.uri.getHost());
    }

    @Nullable
    public String getDatabase() {
        String path = this.uri.getPath();
        return JdbcUrl.decode(StringUtils.startsWith((CharSequence)path, (CharSequence)"/") ? path.substring(1) : path);
    }

    public String getUsername() {
        return this.username;
    }

    public String getPassword() {
        return this.password;
    }

    @Nonnull
    public JdbcUrl setServerHost(String serverHost) {
        this.uri.setHost(serverHost);
        return this;
    }

    @Nonnull
    public JdbcUrl setDatabase(String database) {
        this.uri.setPath("/" + database);
        return this;
    }

    @Nonnull
    public JdbcUrl setUsername(String username) {
        this.username = username;
        return this;
    }

    @Nonnull
    public JdbcUrl setPassword(String password) {
        this.password = password;
        return this;
    }

    @Nonnull
    public JdbcUrl setPort(int port) {
        this.uri.setPort(port);
        return this;
    }

    public String toString() {
        String url = "jdbc:" + this.uri.toString();
        return JdbcUrl.decode(url);
    }

    private static String encode(@Nonnull String context) {
        try {
            return URLEncoder.encode(context, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new AzureToolkitRuntimeException(e.getMessage());
        }
    }

    private static String decode(@Nonnull String context) {
        try {
            return URLDecoder.decode(context, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new AzureToolkitRuntimeException(e.getMessage());
        }
    }

    public boolean equals(@Nullable Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        JdbcUrl jdbcUrl = (JdbcUrl)o;
        return Objects.equals(this.uri.toString(), jdbcUrl.uri.toString());
    }

    public int hashCode() {
        return Objects.hash(this.uri.toString());
    }

    private static class SQLServerJdbcUrl
    extends JdbcUrl {
        private SQLServerJdbcUrl(String url) {
            super(StringUtils.replaceOnce((String)url, (String)";", (String)"?").replaceAll(";", "&"));
        }

        @Override
        int getDefaultPort() {
            return 1433;
        }

        @Override
        @Nonnull
        public String getDefaultDriverClass() {
            return "com.microsoft.sqlserver.jdbc.SQLServerDriver";
        }

        @Override
        @Nonnull
        public JdbcUrl setDatabase(String database) {
            this.uri.setParameter("database", database);
            return this;
        }

        @Override
        @Nullable
        public String getDatabase() {
            return this.uri.getQueryParams().stream().filter(e -> StringUtils.equals((CharSequence)e.getName(), (CharSequence)"database")).map(NameValuePair::getValue).findFirst().orElse(null);
        }

        @Override
        public String toString() {
            String url = "jdbc:" + StringUtils.replaceOnce((String)this.uri.toString(), (String)"?", (String)";").replaceAll("&", ";");
            return JdbcUrl.decode(url);
        }
    }

    private static class PostgreSQLJdbcUrl
    extends JdbcUrl {
        private PostgreSQLJdbcUrl(@Nonnull String url) {
            super(url);
        }

        @Override
        int getDefaultPort() {
            return 5432;
        }

        @Override
        @Nonnull
        public String getDefaultDriverClass() {
            return "org.postgresql.Driver";
        }
    }

    private static class MySQLJdbcUrl
    extends JdbcUrl {
        private MySQLJdbcUrl(@Nonnull String url) {
            super(url);
        }

        @Override
        int getDefaultPort() {
            return 3306;
        }

        @Override
        @Nonnull
        public String getDefaultDriverClass() {
            return "com.mysql.cj.jdbc.Driver";
        }
    }
}

