/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 */

package com.microsoft.azure.toolkit.lib.storage.share;

import com.azure.storage.file.share.ShareClient;
import com.azure.storage.file.share.ShareServiceClient;
import com.microsoft.azure.toolkit.lib.common.action.Action;
import com.microsoft.azure.toolkit.lib.common.action.AzureActionManager;
import com.microsoft.azure.toolkit.lib.common.bundle.AzureString;
import com.microsoft.azure.toolkit.lib.common.exception.AzureToolkitRuntimeException;
import com.microsoft.azure.toolkit.lib.common.messager.AzureMessager;
import com.microsoft.azure.toolkit.lib.common.messager.IAzureMessager;
import com.microsoft.azure.toolkit.lib.common.model.AzResource;
import com.microsoft.azure.toolkit.lib.common.operation.AzureOperation;
import com.microsoft.azure.toolkit.lib.storage.model.StorageFile;
import lombok.Getter;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;

public class ShareDraft extends Share implements AzResource.Draft<Share, ShareClient> {
    @Getter
    @Nullable
    private final Share origin;

    ShareDraft(@Nonnull String name, @Nonnull ShareModule module) {
        super(name, module);
        this.origin = null;
    }

    ShareDraft(@Nonnull Share origin) {
        super(origin);
        this.origin = origin;
    }

    @Override
    public void reset() {
        // do nothing
    }

    @Nonnull
    @Override
    @AzureOperation(name = "azure/storage.create_share.share", params = {"this.getName()"})
    public ShareClient createResourceInAzure() {
        final ShareModule module = (ShareModule) this.getModule();
        final ShareServiceClient client = module.getFileShareServiceClient();
        if (Objects.isNull(client)) {
            throw new AzureToolkitRuntimeException(String.format("Failed to create Share (%s) because storage Account (%s) doesn't exist.", this.getName(), module.getParent().getName()));
        }
        final IAzureMessager messager = AzureMessager.getMessager();
        messager.info(AzureString.format("Start creating File Share ({0}).", this.getName()));
        final ShareClient share = client.createShare(this.getName());
        final Action<StorageFile> createFile = AzureActionManager.getInstance().getAction(CREATE_FILE).bind(this);
        final Action<StorageFile> createDir = AzureActionManager.getInstance().getAction(CREATE_DIRECTORY).bind(this);
        final Action<StorageFile> upload = AzureActionManager.getInstance().getAction(UPLOAD_FILES).bind(this);
        messager.success(AzureString.format("File Share ({0}) is successfully created.", this.getName()), createFile, createDir, upload);
        return share;
    }

    @Nonnull
    @Override
    @AzureOperation(name = "azure/storage.update_share.share", params = {"this.getName()"})
    public ShareClient updateResourceInAzure(@Nonnull ShareClient origin) {
        throw new AzureToolkitRuntimeException("not supported");
    }

    @Override
    public boolean isModified() {
        return false;
    }
}
