/*
 * The MIT License (MIT)
 * Copyright (c) 2018 Microsoft Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.microsoft.azure.documentdb.bulkexecutor;

import java.time.Duration;
import java.util.Collections;
import java.util.List;

public class BulkDeleteResponse {
    /**
     * Total number of documents deleted.
     */
    final private int numberOfDocumentsDeleted;

    /**
     * Total request units consumed.
     */
    final private double totalRequestUnitsConsumed;

    /**
     * Total bulk delete time.
     */
    final private Duration totalTimeTaken;

    /**
     * The list of failures which surfaced out.
     */
    final private List<Exception> failures;
    
    /**
     * The list of deletes which failed along with the specific exceptions leading to the failures
     */
    final private List<BulkDeleteFailure> failedDeletes;

    public BulkDeleteResponse(
            int numberOfDocumentsDeleted,
            double totalRequestUnitsConsumed,
            Duration totalTimeTaken,
            List<Exception> failures,
            List<BulkDeleteFailure> failedDeletes) {
        this.numberOfDocumentsDeleted = numberOfDocumentsDeleted;
        this.totalRequestUnitsConsumed = totalRequestUnitsConsumed;
        this.totalTimeTaken = totalTimeTaken;
        this.failures = failures;
        this.failedDeletes = failedDeletes;
    }

    /**
     * Gets number of documents successfully deleted.
     *
     * <p> If this number is less than actual batch size (meaning some documents failed to get deleted),
     * use {@link #getErrors()} to get the failure cause.
     * @return the numberOfDocumentsDeleted
     */
    public int getNumberOfDocumentsDeleted() {
        return numberOfDocumentsDeleted;
    }

    /**
     * Gets the total request units consumed during the bulk delete.
     *
     * @return the totalRequestUnitsConsumed
     */
    public double getTotalRequestUnitsConsumed() {
        return totalRequestUnitsConsumed;
    }

    /**
     * Gets the total time taken for bulk delete.
     *
     * @return the totalTimeTaken
     */
    public Duration getTotalTimeTaken() {
        return totalTimeTaken;
    }

    /**
     * Gets failure list if some documents failed to get deleted.
     *
     * @return list of errors or empty list if no error.
     */
    public List<Exception> getErrors() {
        return Collections.unmodifiableList(failures);
    }
    
    /**
     * Gets the list of failures during the bulk delete execution
     * @return
     */
    public List<BulkDeleteFailure> getFailedDeletes() {
        return Collections.unmodifiableList(failedDeletes);
    }
}
