/*
 * The MIT License (MIT)
 * Copyright (c) 2017 Microsoft Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.microsoft.azure.documentdb.bulkexecutor;

import java.time.Duration;
import java.util.Collections;
import java.util.List;

public class BulkImportResponse {
    /**
     * Total number of documents imported.
     */
    final private int numberOfDocumentsImported;

    /**
     * Total request units consumed.
     */
    final private double totalRequestUnitsConsumed;

    /**
     * Total bulk import time.
     */
    final private Duration totalTimeTaken;

    /**
     * The list of failures which surfaced out.
     */
    final private List<Exception> failures;

    /**
     * The list of bad input documents which caused failures.
     */
    final private List<Object> badInputDocuments;
    
    /**
     * The list of imports which failed along with the specific exceptions leading to the failures
     */
    final private List<BulkImportFailure> failedImports;

    BulkImportResponse(int numberOfDocumentsImported,
            double totalRequestUnitsConsumed,
            Duration totalTimeTaken,
            List<Exception> failures,
            List<Object> badInputDocuments,
            List<BulkImportFailure> failedImports) {
        this.numberOfDocumentsImported = numberOfDocumentsImported;
        this.totalRequestUnitsConsumed = totalRequestUnitsConsumed;
        this.totalTimeTaken = totalTimeTaken;
        this.failures = failures;
        this.badInputDocuments = badInputDocuments;
        this.failedImports = failedImports;
    }

    /**
     * Gets number of documents successfully inserted.
     *
     * <p> If this number is less than actual batch size (meaning some documents failed to get inserted),
     * use {@link #getErrors()} to get the failure cause.
     * @return the numberOfDocumentsImported
     */
    public int getNumberOfDocumentsImported() {
        return numberOfDocumentsImported;
    }

    /**
     * Gets the total request units consumed during the bulk import.
     * 
     * @return the totalRequestUnitsConsumed
     */
    public double getTotalRequestUnitsConsumed() {
        return totalRequestUnitsConsumed;
    }

    /**
     * Gets the total time taken for bulk import.
     * 
     * @return the totalTimeTaken
     */
    public Duration getTotalTimeTaken() {
        return totalTimeTaken;
    }

    /**
     * Gets failure list if some documents failed to get inserted.
     *
     * @return list of errors or empty list if no error.
     */
    public List<Exception> getErrors() {
        return Collections.unmodifiableList(failures);
    }

    /**
     * Gets the list of bad input documents which caused failures.
     *
     * @return list of bad input documents.
     */
    public List<Object> getBadInputDocuments() {
        return Collections.unmodifiableList(badInputDocuments);
    }
    
    /**
     * Gets the list of failed imports during the bulk import
     * @return
     */
    public List<BulkImportFailure> getFailedImports() {
        return Collections.unmodifiableList(failedImports);
    }
}
