/*
 * The MIT License (MIT)
 * Copyright (c) 2017 Microsoft Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.microsoft.azure.documentdb.bulkexecutor.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

final class PathParser
{
    private final static char SEGMENT_SEPARATOR = '/';
    private final static String ERROR_MESSAGE_FORMAT = "Invalid path \"%s\", failed at %d";

    /**
     * Extract parts from a given path for '/' as the separator.
     * <p>
     * This code doesn't do as much validation as the backend, as it assumes that IndexingPolicy path coming from the backend is valid.
     *
     * @param path specifies a partition key given as a path.
     * @return a list of all the parts for '/' as the separator.
     */
    public static List<String> getPathParts(String path)
    {
        List<String> tokens = new ArrayList<String>();
        AtomicInteger currentIndex = new AtomicInteger();

        while (currentIndex.get() < path.length())
        {
            char currentChar = path.charAt(currentIndex.get());
            if (currentChar != SEGMENT_SEPARATOR)
            {
                throw new IllegalArgumentException(
                    String.format(ERROR_MESSAGE_FORMAT, path, currentIndex.get()));
            }

            if (currentIndex.incrementAndGet() == path.length())
            {
                break;
            }

            currentChar = path.charAt(currentIndex.get());
            if (currentChar == '\"' || currentChar == '\'')
            {
                // Handles the partial path given in quotes such as "'abc/def'"
                tokens.add(getEscapedToken(path, currentIndex));
            }
            else
            {
                tokens.add(getToken(path, currentIndex));
            }
        }

        return tokens;
    }

    private static String getEscapedToken(String path, AtomicInteger currentIndex)
    {
        char quote = path.charAt(currentIndex.get());
        int newIndex = currentIndex.incrementAndGet();

        while (true)
        {
            newIndex = path.indexOf(quote, newIndex);
            if (newIndex == -1)
            {
                throw new IllegalArgumentException(
                    String.format(ERROR_MESSAGE_FORMAT, path, currentIndex.get()));
            }

            // Ignore escaped quote in the partial path we look at such as "'abc/def \'12\'/ghi'"
            if (path.charAt(newIndex - 1) != '\\')
            {
                break;
            }

            ++newIndex;
        }

        String token = path.substring(currentIndex.get(), newIndex);
        currentIndex.set(newIndex + 1);

        return token;
    }

    private static String getToken(String path, AtomicInteger currentIndex)
    {
        int newIndex = path.indexOf(SEGMENT_SEPARATOR, currentIndex.get());
        String token = null;
        if (newIndex == -1)
        {
            token = path.substring(currentIndex.get());
            currentIndex.set(path.length());
        }
        else
        {
            token = path.substring(currentIndex.get(), newIndex);
            currentIndex.set(newIndex);
        }

        token = token.trim();
        return token;
    }
}