// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

package com.microsoft.azure.javamsalruntime;

/**
 * Represents an MSALRuntime account in a way that MSAL Java can use to create its own Account
 * object <p> Uses the APIs defined in MSALRuntimeAccount.h to retrieve data from a single
 * MSALRUNTIME_ACCOUNT_HANDLE
 */
public class Account implements AutoCloseable {
    private AccountHandle handle;
    private String accountId;
    private String accountClientInfo;

    Account(AccountHandle accountHandle) {
        if (accountHandle.isHandleValid()) {
            this.handle = accountHandle;
        }
    }

    AccountHandle getHandle() {
        return handle;
    }

    /**
     * Retrieves the account ID from MSALRuntime, and stores it in this Account instance
     */
    public String getAccountId() {
        if (accountId == null) {
            accountId = HandleBase.getString(
                    handle,
                    (accountHandle, id, bufferSize)
                            -> MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetAccountId(
                                    accountHandle.value(), id, bufferSize));
        }

        return accountId;
    }

    /**
     * Retrieves the client info String from MSALRuntime, and stores it in this Account instance
     */
    public String getClientInfo() {
        if (accountClientInfo == null) {
            accountClientInfo = HandleBase.getString(
                    handle,
                    (accountHandle, clientInfo, bufferSize)
                            -> MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetClientInfo(
                                    accountHandle.value(), clientInfo, bufferSize));
        }

        return accountClientInfo;
    }

    @Override
    public void close() {
        if (handle != null) {
            handle.close();

            handle = null;
        }
    }
}
