// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

package com.microsoft.azure.javamsalruntime;

import com.sun.jna.Structure;
import com.sun.jna.ptr.IntByReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Contains all of the behavior needed to parse the result of calls to MSALRuntime's SignIn and
 * AcquireToken APIs, and to complete an associated AsyncHandler using that parsed result <p> This
 * AuthResult class is intended to be easily translated to MSAL Java's AuthenticationResult class,
 * but AuthResult should never be required to map directly to an AuthenticationResult or vice-versa
 * <p>
 * An internal AuthResultCallback class contains a callback method, which MSALRuntime will call when
 * a result is ready to be parsed
 */
@Structure.FieldOrder({"unused"})
public class AuthResult extends Structure {
    // JNA requires at least one public field to convert its Structure class to a native structure
    public int unused;

    private static final Logger LOG = LoggerFactory.getLogger(AuthResult.class);

    private AuthResultHandle resultHandle;
    private String idToken;
    private String accessToken;
    private long accessTokenExpirationTime;
    private Account account;
    private String authorizationHeader;
    private Boolean isPopAuthorization;

    public AuthResult(AuthResultHandle authResultHandle) {
        this.resultHandle = authResultHandle;
        parseAuthResult();
    }

    public String getIdToken() {
        return idToken;
    }

    public String getAccessToken() {
        return accessToken;
    }

    public long getAccessTokenExpirationTime() {
        return accessTokenExpirationTime;
    }

    public Account getAccount() {
        return account;
    }

    /**
     * Returns the authorization header, used to retrieve a proof-of-possession token
     */
    public String getAuthorizationHeader() {
        if (authorizationHeader == null) {
            authorizationHeader = HandleBase.getString(
                    this.resultHandle,
                    (error, authHeader, bufferSize)
                            -> MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetAuthorizationHeader(
                                    this.resultHandle, authHeader, bufferSize));
        }

        return authorizationHeader;
    }

    /**
     * Returns true if this authResult represents a proof-of-possession authorization
     */
    public boolean isPopAuthorization() {
        if (isPopAuthorization == null) {
            IntByReference isPop = new IntByReference(0);
            MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(
                    MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_IsPopAuthorization(
                            this.resultHandle, isPop));

            // MSALRUNTIME_IsPopAuthorization uses bool_t, which is an alias for an int as per
            // MSALRuntimeTypes.h, and passing an int pointer result in 1=true/0=false
            isPopAuthorization = isPop.getValue() == 1;
        }

        return isPopAuthorization;
    }

    /**
     * Calls various MSALRuntime APIs to retrieve data using the MSALRUNTIME_AUTH_RESULT_HANDLE
     * passed into the callback method
     */
    void parseAuthResult() {
        if (this.resultHandle.isHandleValid()) {
            LOG.info("Checking auth result error.");
            ErrorHandle error = new ErrorHandle();

            MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(
                    MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetError(
                            this.resultHandle, error));

            MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(error);

            LOG.info("Parsing auth result.");
            isPopAuthorization();
            parseAndSetAccessToken();
            parseAndSetIdToken();
            parseAndSetAccount();
        } else {
            throw new MsalInteropException(
                    "Auth result handle was invalid, could not parse.", "msalruntime_error");
        }
    }

    /**
     * If the auth result handle has an access token, retrieve and store it in this AuthResult
     */
    void parseAndSetAccessToken() {
        if (isPopAuthorization) {
            // POP tokens are returned as part of the authorization header,
            //   and is formatted as "pop {the signed access token}"
            this.accessToken = getAuthorizationHeader().split(" ")[1];
        } else {
            // If it is not a POP token, just get the token from the GetAccessToken API
            this.accessToken = HandleBase.getString(
                    resultHandle,
                    (authResultHandle, accessToken, bufferSize)
                            -> MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetAccessToken(
                                    this.resultHandle, accessToken, bufferSize));
        }
    }

    /**
     * If the auth result handle has an id token, retrieve and store it in this AuthResult
     */
    void parseAndSetIdToken() {
        this.idToken = HandleBase.getString(
                resultHandle,
                (authResultHandle, rawIdToken, bufferSize)
                        -> MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetRawIdToken(
                                this.resultHandle, rawIdToken, bufferSize));
    }

    /**
     * If the auth result handle has an account handle, parse and store it in this AuthResult
     */
    void parseAndSetAccount() {
        AccountHandle accountHandle = new AccountHandle();

        MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(
                MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetAccount(
                        this.resultHandle, accountHandle));

        this.account = new Account(accountHandle);
    }
}
