// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

package com.microsoft.azure.javamsalruntime;

import com.sun.jna.Callback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Contains any functionality needed to allow callbacks sent to MSALRuntime to complete futures
 * <p>
 * These futures will be returned to the MSAL Java Brokers layer, and chained to a future that was
 * created by MSAL Java and returned as part of an acquire token call <p> Once this future is
 * completed in a callback, the future it was chained will convert the interop layer's result into
 * an MSAL Java AuthenticationResult, and complete the future like a normal acquire token call
 */
public class MsalRuntimeFuture extends CompletableFuture<Object> {
    private static final Logger LOG = LoggerFactory.getLogger(MsalRuntimeFuture.class);

    // This table and counter are used to keep track of uncompleted futures, and offer a way for the
    // callback methods to find the right future to complete
    static HashMap<Integer, MsalRuntimeFuture> msalRuntimeFutures = new HashMap<>();
    static AtomicInteger asyncHandleCounter = new AtomicInteger();

    AsyncHandle handle = new AsyncHandle();
    // We only know a future is ready to complete when MSALRuntime calls the associated callback
    // reference,
    //  so keeping a reference to the object here will prevent premature garbage collection of that
    //  callback
    Callback callback;

    // Generate unique keys, used by callback methods to identify the correct future to complete
    final Integer msalRuntimeFuturesKey = asyncHandleCounter.incrementAndGet();

    public MsalRuntimeFuture(Callback callback) {
        this.callback = callback;

        // Add this future to handles table, so the callback methods can find and complete them
        msalRuntimeFutures.putIfAbsent(this.msalRuntimeFuturesKey, this);
    }

    /**
     * Calls MSALRuntime's cancel API to cancel the related async operation, and releases the async
     * handle
     */
    public void cancelAsyncOperation() {
        if (handle.isHandleValid()) {
            try {
                LOG.info("Canceling async operation.");
                // Tells MSALRuntime to cancel the async operation, and releases the handle
                MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(
                        MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_CancelAsyncOperation(
                                handle));
            } catch (MsalInteropException msalInteropEx) {
                throw msalInteropEx;
            } catch (Exception e) {
                MsalRuntimeInterop.ERROR_HELPER.logUnknownErrorReleasingHandle(e);
            } finally {
                handle.release();

                // Set local handle to null, to indicate that it has been released and prevent
                // attempts to use it again
                this.handle = null;
            }
        }
    }
}
