// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

package com.microsoft.azure.javamsalruntime;

import com.sun.jna.Structure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Contains all of the behavior needed to parse the result of calls to MSALRuntime's ReadAccount
 * APIs, and to complete an associated AsyncHandler using that parsed result <p> This
 * ReadAccountResult class is intended to be easily translated to MSAL Java's Account class, but
 * ReadAccountResult should never be required to map directly to an Account or vice-versa <p> An
 * internal ReadAccountResultCallback class contains a callback method, which MSALRuntime will call
 * when a result is ready to be parsed
 */
@Structure.FieldOrder({"unused"})
public class ReadAccountResult extends Structure {
    private static final Logger LOG = LoggerFactory.getLogger(ReadAccountResult.class);

    // JNA requires at least one public field to convert its Structure class to a native structure
    public int unused;

    private Account account;
    private ReadAccountResultHandle resultHandle;

    public ReadAccountResult(ReadAccountResultHandle readAccountResultHandle) {
        this.resultHandle = readAccountResultHandle;
        parseReadAccountResult();
    }

    public Account getAccount() {
        return account;
    }

    /**
     * Calls various MSALRuntime APIs to retrieve data using the
     * MSALRUNTIME_READ_ACCOUNT_RESULT_HANDLE passed into the callback method
     */
    void parseReadAccountResult() {
        if (this.resultHandle.isHandleValid()) {
            LOG.info("Checking read account result error.");
            ErrorHandle error = new ErrorHandle();

            MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(
                    MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetReadAccountError(
                            this.resultHandle, error));

            MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(error);

            LOG.info("Parsing read account result.");
            parseAndSetAccount();
        } else {
            throw new MsalInteropException(
                    "Read account result handle was invalid, could not parse.", "msalruntime_error");
        }
    }

    /**
     * If the auth result handle has an account handle, parse and store it in this ReadAccountResult
     */
    void parseAndSetAccount() {
        AccountHandle accountHandle = new AccountHandle();

        MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(
                MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetReadAccount(
                        this.resultHandle, accountHandle));

        this.account = new Account(accountHandle);
    }
}
