// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

package com.microsoft.azure.javamsalruntime;

import com.sun.jna.Structure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Contains all of the behavior needed to parse the result of calls to MSALRuntime's SignOut APIs,
 * and to complete an associated AsyncHandler using that parsed result <p> An internal
 * SignOutResultCallback class contains a callback method, which MSALRuntime will call when a result
 * is ready to be parsed
 */
@Structure.FieldOrder({"unused"})
public class SignOutResult extends Structure {
    // JNA requires at least one public field to convert its Structure class to a native structure
    public int unused;

    private static final Logger LOG = LoggerFactory.getLogger(SignOutResult.class);

    private SignOutResultHandle resultHandle;

    public SignOutResult(SignOutResultHandle signOutResultHandle) {
        this.resultHandle = signOutResultHandle;
        parseSignOutResult();
    }

    /**
     * Calls various MSALRuntime APIs to retrieve data using the MSALRUNTIME_SIGNOUT_RESULT_HANDLE
     * passed into the callback method <p> NOTE: Currently, there is nothing for us to parse from a
     * sign out result <p> This is just here to complete the mapping to MSALRuntime's SignOut API in
     * a way that's consistent with the other result types, and to allow easy addition of any future
     * parsing
     */
    void parseSignOutResult() {
        if (this.resultHandle.isHandleValid()) {
            LOG.info("Checking sign out result error.");
            ErrorHandle error = new ErrorHandle();

            MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(
                    MsalRuntimeInterop.MSALRUNTIME_LIBRARY.MSALRUNTIME_GetSignOutError(
                            this.resultHandle, error));

            MsalRuntimeInterop.ERROR_HELPER.checkMsalRuntimeError(error);

            LOG.info("Parsing sign out result.");
            // Nothing for us to actually parse, this is just for consistency with the other result
            // classes
        } else {
            throw new MsalInteropException(
                    "Sign out result handle was invalid, could not parse.", "msalruntime_error");
        }
    }
}
