/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.sdk.iot.service.messaging;

import com.azure.core.credential.AzureSasCredential;
import com.azure.core.credential.TokenCredential;
import com.microsoft.azure.sdk.iot.service.auth.IotHubConnectionStringBuilder;
import com.microsoft.azure.sdk.iot.service.exceptions.IotHubException;
import com.microsoft.azure.sdk.iot.service.messaging.ErrorContext;
import com.microsoft.azure.sdk.iot.service.messaging.IotHubServiceClientProtocol;
import com.microsoft.azure.sdk.iot.service.messaging.Message;
import com.microsoft.azure.sdk.iot.service.messaging.MessagingClientOptions;
import com.microsoft.azure.sdk.iot.service.messaging.SendResult;
import com.microsoft.azure.sdk.iot.service.messaging.Tools;
import com.microsoft.azure.sdk.iot.service.transport.TransportUtils;
import com.microsoft.azure.sdk.iot.service.transport.amqps.CloudToDeviceMessageConnectionHandler;
import com.microsoft.azure.sdk.iot.service.transport.amqps.ReactorRunner;
import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MessagingClient {
    private static final Logger log = LoggerFactory.getLogger(MessagingClient.class);
    private static final int START_REACTOR_TIMEOUT_MILLISECONDS = 60000;
    private static final int STOP_REACTOR_TIMEOUT_MILLISECONDS = 5000;
    private static final int MESSAGE_SEND_TIMEOUT_MILLISECONDS = 60000;
    private final Consumer<ErrorContext> errorProcessor;
    private final CloudToDeviceMessageConnectionHandler cloudToDeviceMessageConnectionHandler;
    private ReactorRunner reactorRunner;
    private final String hostName;

    public MessagingClient(String connectionString, IotHubServiceClientProtocol protocol) {
        this(connectionString, protocol, MessagingClientOptions.builder().build());
    }

    public MessagingClient(String connectionString, IotHubServiceClientProtocol protocol, MessagingClientOptions options) {
        if (Tools.isNullOrEmpty(connectionString).booleanValue()) {
            throw new IllegalArgumentException(connectionString);
        }
        if (options == null) {
            throw new IllegalArgumentException("MessagingClientOptions cannot be null for this constructor");
        }
        this.errorProcessor = options.getErrorProcessor();
        this.hostName = IotHubConnectionStringBuilder.createIotHubConnectionString(connectionString).getHostName();
        this.cloudToDeviceMessageConnectionHandler = new CloudToDeviceMessageConnectionHandler(connectionString, protocol, this.errorProcessor, options.getProxyOptions(), options.getSslContext(), options.getKeepAliveInterval());
        MessagingClient.commonConstructorSetup();
    }

    public MessagingClient(String hostName, TokenCredential credential, IotHubServiceClientProtocol protocol) {
        this(hostName, credential, protocol, MessagingClientOptions.builder().build());
    }

    public MessagingClient(String hostName, TokenCredential credential, IotHubServiceClientProtocol protocol, MessagingClientOptions options) {
        Objects.requireNonNull(credential);
        if (Tools.isNullOrEmpty(hostName).booleanValue()) {
            throw new IllegalArgumentException("HostName cannot be null or empty");
        }
        if (options == null) {
            throw new IllegalArgumentException("MessagingClientOptions cannot be null for this constructor");
        }
        if (options.getProxyOptions() != null && protocol != IotHubServiceClientProtocol.AMQPS_WS) {
            throw new UnsupportedOperationException("Proxies are only supported over AMQPS_WS");
        }
        this.errorProcessor = options.getErrorProcessor();
        this.hostName = hostName;
        this.cloudToDeviceMessageConnectionHandler = new CloudToDeviceMessageConnectionHandler(hostName, credential, protocol, this.errorProcessor, options.getProxyOptions(), options.getSslContext(), options.getKeepAliveInterval());
        MessagingClient.commonConstructorSetup();
    }

    public MessagingClient(String hostName, AzureSasCredential azureSasCredential, IotHubServiceClientProtocol protocol) {
        this(hostName, azureSasCredential, protocol, MessagingClientOptions.builder().build());
    }

    public MessagingClient(String hostName, AzureSasCredential azureSasCredential, IotHubServiceClientProtocol protocol, MessagingClientOptions options) {
        Objects.requireNonNull(azureSasCredential);
        Objects.requireNonNull(options);
        if (options.getProxyOptions() != null && protocol != IotHubServiceClientProtocol.AMQPS_WS) {
            throw new UnsupportedOperationException("Proxies are only supported over AMQPS_WS");
        }
        this.errorProcessor = options.getErrorProcessor();
        this.hostName = hostName;
        this.cloudToDeviceMessageConnectionHandler = new CloudToDeviceMessageConnectionHandler(hostName, azureSasCredential, protocol, this.errorProcessor, options.getProxyOptions(), options.getSslContext(), options.getKeepAliveInterval());
        MessagingClient.commonConstructorSetup();
    }

    private static void commonConstructorSetup() {
        log.debug("Initialized a MessagingClient instance using SDK version {}", (Object)TransportUtils.serviceVersion);
    }

    public synchronized void open() throws IotHubException, IOException, InterruptedException, TimeoutException {
        this.open(60000);
    }

    public synchronized void open(int timeoutMilliseconds) throws IotHubException, IOException, InterruptedException, TimeoutException {
        boolean timedOut;
        if (this.isOpen()) {
            return;
        }
        if (timeoutMilliseconds < 0) {
            throw new IllegalArgumentException("timeoutMilliseconds must be greater than or equal to 0");
        }
        AtomicReference<Object> iotHubException = new AtomicReference<Object>(null);
        AtomicReference<Object> ioException = new AtomicReference<Object>(null);
        log.debug("Opening MessagingClient");
        this.reactorRunner = new ReactorRunner(this.hostName, "MessagingClient", this.cloudToDeviceMessageConnectionHandler);
        CountDownLatch openLatch = new CountDownLatch(1);
        this.cloudToDeviceMessageConnectionHandler.setOnConnectionOpenedCallback(openLatch::countDown);
        new Thread(() -> {
            try {
                this.reactorRunner.run();
                log.trace("MessagingClient Amqp reactor stopped, checking that the connection was opened");
                this.cloudToDeviceMessageConnectionHandler.verifyConnectionWasOpened();
                log.trace("MessagingClient reactor did successfully open the connection, returning without exception");
            }
            catch (IOException e) {
                ioException.set(e);
            }
            catch (IotHubException e) {
                iotHubException.set(e);
            }
            finally {
                openLatch.countDown();
            }
        }).start();
        boolean bl = timedOut = !openLatch.await(timeoutMilliseconds, TimeUnit.MILLISECONDS);
        if (timedOut) {
            throw new TimeoutException("Timed out waiting for the connection to the service to open");
        }
        if (ioException.get() != null) {
            throw (IOException)ioException.get();
        }
        if (iotHubException.get() != null) {
            throw (IotHubException)iotHubException.get();
        }
        log.info("Opened MessagingClient");
    }

    public synchronized void close() throws InterruptedException {
        this.close(5000);
    }

    public synchronized void close(int timeoutMilliseconds) throws InterruptedException {
        if (this.reactorRunner == null) {
            return;
        }
        if (timeoutMilliseconds < 0) {
            throw new IllegalArgumentException("timeoutMilliseconds must be greater than or equal to 0");
        }
        this.reactorRunner.stop(timeoutMilliseconds);
        this.reactorRunner = null;
        log.info("Closed MessagingClient");
    }

    public void send(String deviceId, Message message) throws IotHubException, InterruptedException, TimeoutException, IllegalStateException {
        this.send(deviceId, null, message, 60000);
    }

    public void send(String deviceId, Message message, int timeoutMilliseconds) throws IotHubException, InterruptedException, TimeoutException, IllegalStateException {
        this.send(deviceId, null, message, timeoutMilliseconds);
    }

    public void send(String deviceId, String moduleId, Message message) throws IotHubException, InterruptedException, TimeoutException, IllegalStateException {
        this.send(deviceId, moduleId, message, 60000);
    }

    public void send(String deviceId, String moduleId, Message message, int timeoutMilliseconds) throws IotHubException, InterruptedException, TimeoutException, IllegalStateException {
        if (timeoutMilliseconds < 0) {
            throw new IllegalArgumentException("timeoutMilliseconds must be greater than or equal to 0");
        }
        AtomicReference exception = new AtomicReference();
        CountDownLatch messageSentLatch = new CountDownLatch(1);
        Consumer<SendResult> onMessageAcknowledgedCallback = sendResult -> {
            if (sendResult.wasSentSuccessfully()) {
                log.trace("Message acknowledged callback executed for cloud to device message with correlation id {} that was successfully sent.", (Object)sendResult.getCorrelationId());
            } else {
                log.trace("Message acknowledged callback executed for cloud to device message with correlation id {} that failed to send.", (Object)sendResult.getCorrelationId());
                exception.set(sendResult.getException());
            }
            messageSentLatch.countDown();
        };
        this.sendAsync(deviceId, moduleId, message, onMessageAcknowledgedCallback, null);
        if (timeoutMilliseconds == 0) {
            messageSentLatch.await();
        } else {
            boolean timedOut;
            boolean bl = timedOut = !messageSentLatch.await(timeoutMilliseconds, TimeUnit.MILLISECONDS);
            if (timedOut) {
                throw new TimeoutException("Timed out waiting for message to be acknowledged");
            }
        }
        if (exception.get() != null) {
            throw (IotHubException)exception.get();
        }
    }

    public void sendAsync(String deviceId, Message message, Consumer<SendResult> onMessageSentCallback, Object context) throws IllegalStateException {
        this.sendAsync(deviceId, null, message, onMessageSentCallback, context);
    }

    public void sendAsync(String deviceId, String moduleId, Message message, Consumer<SendResult> onMessageSentCallback, Object context) throws IllegalStateException {
        if (!this.isOpen()) {
            throw new IllegalStateException("Client must be opened before any message can be sent");
        }
        if (moduleId == null) {
            log.info("Sending cloud to device message with correlation id {}", (Object)message.getCorrelationId());
        } else {
            log.info("Sending cloud to module message with correlation id {}", (Object)message.getCorrelationId());
        }
        this.cloudToDeviceMessageConnectionHandler.sendAsync(deviceId, moduleId, message, onMessageSentCallback, context);
    }

    public boolean isOpen() {
        return this.reactorRunner != null && this.reactorRunner.isRunning();
    }
}

