//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.lang.AutoCloseable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.SpeechConfig;

/**
 * Represents the speaker verification model used for speaker recognition.
 * Note: close() must be called in order to release underlying resources held by the object.
 * 
 */
public class SpeakerVerificationModel implements AutoCloseable
{
    // load the native library.
    static {
        @SuppressWarnings("unused")
        Class<?> speechFactorLoadTrigger = SpeechConfig.speechConfigClass;
    }

    /**
     * Creates a speaker verification model from a voice profile.
     * @param profile A voice profile.
     * @return The speaker verification model being created.
     */
    public static SpeakerVerificationModel fromProfile(VoiceProfile profile) {        
        Contracts.throwIfNull(profile, "profile cannot be null");
        IntRef modelHandle = new IntRef(0);
        Contracts.throwIfFail(createSpeakerVerificationModel(modelHandle, profile.getImpl()));
        return new SpeakerVerificationModel(modelHandle);
    }

    /**
     * Dispose of associated resources.
     */
    @Override
    public void close() {
        if (disposed) {
            return;
        }
        if (speakerVerificationModelHandle != null) {
            speakerVerificationModelHandle.close();
            speakerVerificationModelHandle = null;    
        }
        disposed = true;
    }

    SpeakerVerificationModel(IntRef modelHandle) {
        Contracts.throwIfNull(modelHandle, "modelHandle");
        speakerVerificationModelHandle = new SafeHandle(modelHandle.getValue(), SafeHandleType.SpeakerVerificationModel);
    }

    /*! \cond INTERNAL */

    /**
     * Returns the speaker verification model.
     * @return The implementation of the model.
     */
    public SafeHandle getImpl() {
        return speakerVerificationModelHandle;
    }
    
    /*! \endcond */

    private boolean disposed = false;
    private SafeHandle speakerVerificationModelHandle = null;
    private final static native long createSpeakerVerificationModel(IntRef modelHandle, SafeHandle profileHandle);
}
