//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.lang.AutoCloseable;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.ResultReason;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;

/**
 * Defines result of processing a voice profile.
 */
public final class VoiceProfileResult implements AutoCloseable
{
    VoiceProfileResult(long result) {
        Contracts.throwIfNull(result, "result");
        this.resultHandle = new SafeHandle(result, SafeHandleType.VoiceProfileResult);

        StringRef stringVal = new StringRef("");
        Contracts.throwIfFail(getResultId(resultHandle, stringVal));
        this.resultId = stringVal.getValue();
                
        IntRef intVal = new IntRef(0);
        Contracts.throwIfFail(getResultReason(resultHandle, intVal));
        this.reason = ResultReason.values()[(int)intVal.getValue()];
        
        IntRef propertyHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromResult(resultHandle, propertyHandle));
        this.properties = new PropertyCollection(propertyHandle);
    }

    /**
     * Presents the result identifier.
     * @return The the result identifier.
     */
    public String getResultId() {  
        return this.resultId;
    }

    /**
     * Specifies the reason of voice profile result.
     * @return The reason of the result.
     */
    public ResultReason getReason() {
        return this.reason;
    }

    /**  
     * The set of properties exposed in the result.
     * @return The set of properties exposed in the result.
     */
    public PropertyCollection getProperties() {
        return this.properties;
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public void close() {
        if (this.resultHandle != null) {
            resultHandle.close();
            this.resultHandle = null;
        }

        if (this.properties != null) {
            this.properties.close();
            this.properties = null;
        }
    }

    /**
     * Returns a String that represents the voice profile result.
     * @return A String that represents the voice profile result.
     */
    @Override
    public String toString() {
        return "ResultId:" + getResultId() +
                " Reason:" + getReason() +
                " Json:" + properties.getProperty(PropertyId.SpeechServiceResponse_JsonResult);
    }

    /*! \cond INTERNAL */
    /**
     * Returns the voice profile result implementation.
     * @return The implementation of the result.
     */
    public SafeHandle getImpl() {
        Contracts.throwIfNull(this.resultHandle, "result");
        return this.resultHandle;
    }
    /*! \endcond */

    private SafeHandle resultHandle = null;
    private PropertyCollection properties = null;
    private String resultId = "";
    private ResultReason reason;

    private final native long getResultId(SafeHandle resultHandle, StringRef resultId);
    private final native long getResultReason(SafeHandle resultHandle, IntRef reasonVal);
    private final native long getPropertyBagFromResult(SafeHandle resultHandle, IntRef propertyHandle);
}
