//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.intent;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.KeyedItem;

/**
*  Represents a pattern matching intent used for intent recognition.
*/
public final class PatternMatchingIntent implements KeyedItem {
    private String id;

    /**
    *  Phrases and patterns that will trigger this intent. At least one phrase
    *  must exist to be able to apply this intent to an IntentRecognizer.
    */
    public Collection<String> Phrases;

    /**
    * Creates a pattern matching intent using the specified intent ID.
    * @param intentId A string that represents a unique ID for this intent.
    */
    public PatternMatchingIntent(String intentId) {
        id = intentId;
        Phrases = new ArrayList<>();
    }

    /**
    * Creates a pattern matching intent using the specified intent ID.
    * @param intentId A string that represents a unique ID for this intent.
    * @param phrases A collection of strings used to populate the intent.
    */
    public PatternMatchingIntent(String intentId, Collection<String> phrases) {
        id = intentId;
        Phrases = phrases;
    }

    /**
    * Creates a pattern matching intent using the specified intent ID.
    * @param intentId A string that represents a unique ID for this intent.
    * @param phrases A collection of strings used to populate the intent.
    */
    public PatternMatchingIntent(String intentId, String... phrases) {
        id = intentId;
        Phrases = new ArrayList<String>();
        for(String phrase : phrases)
        {
            Phrases.add(phrase);
        }
    }

    /**
    *  Returns the ID for this item.
    *  @return The ID for this item.
    */
    @Override
    public String getId() {
        return id;
    }

    /**
    *  Sets the ID for this item.
    *  @param id The ID for this item.
    */
    public void setId(String id) {
        this.id = id;
    }
}
