//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.lang.AutoCloseable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.SpeechConfig;

/**
 * Represents the speaker identification model used for speaker recognition.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public class SpeakerIdentificationModel implements AutoCloseable
{
    // load the native library.
    static {
        @SuppressWarnings("unused")
        Class<?> speechFactorLoadTrigger = SpeechConfig.speechConfigClass;
    }

    /**
     * Creates a speaker identification model from the voice profiles.
     * @param profiles A collection of voice profiles.
     * @return The speaker identification model being created.
     */
    public static SpeakerIdentificationModel fromProfiles(List<VoiceProfile> profiles) {

        IntRef modelHandle = new IntRef(0);
        Contracts.throwIfNull(profiles, "profiles cannot be null");
        Contracts.throwIfFail(createSpeakerIdentificationModel(modelHandle));

        for (VoiceProfile profile : profiles) {
            Contracts.throwIfFail(speakerIdentificationModelAddProfile(modelHandle, profile.getImpl()));
        }

        return new SpeakerIdentificationModel(modelHandle);
    }

    /**
     * Dispose of associated resources.
     */
    @Override
    public void close() {
        if (disposed) {
            return;
        }
        if (speakerIdentificationModelHandle != null) {
            speakerIdentificationModelHandle.close();
            speakerIdentificationModelHandle = null;    
        }
        disposed = true;
    }

    SpeakerIdentificationModel(IntRef modelHandle) {
        Contracts.throwIfNull(modelHandle, "modelHandle");
        speakerIdentificationModelHandle = new SafeHandle(modelHandle.getValue(), SafeHandleType.SpeakerIdentificationModel);
    }

    /*! \cond INTERNAL */

    /**
     * Returns the speaker identification model.
     * @return The implementation of the model.
     */
    public SafeHandle getImpl() {
        return speakerIdentificationModelHandle;
    }
    
    /*! \endcond */

    private boolean disposed = false;
    private SafeHandle speakerIdentificationModelHandle = null;
    private final static native long createSpeakerIdentificationModel(IntRef modelHandle);
    private final static native long speakerIdentificationModelAddProfile(IntRef modelHandle, SafeHandle voiceProfile);
}
