//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.util.concurrent.Future;
import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.audio.AudioConfig;
import com.microsoft.cognitiveservices.speech.SpeechConfig;
import com.microsoft.cognitiveservices.speech.speaker.SpeakerRecognitionResult;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.AsyncThreadService;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.speaker.SpeakerIdentificationModel;
import com.microsoft.cognitiveservices.speech.speaker.SpeakerVerificationModel;

/**
 * Performs speaker recognition on the speech input.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class SpeakerRecognizer implements AutoCloseable {

    /**
     * Creates a new instance of a speaker recognizer. If no audio config is provided as input parameter, it will be equivalent to calling with a config 
     * constructed with AudioConfig::FromDefaultMicrophoneInput.
     * @param speechConfig speech configuration.
     * @param audioConfig audio configuration.
     */
    public SpeakerRecognizer(SpeechConfig speechConfig, AudioConfig audioConfig) {

        Contracts.throwIfNull(speechConfig, "speechConfig");
        IntRef handle = new IntRef(0);
        if (audioConfig == null) {
            Contracts.throwIfFail(createFromConfig(handle, speechConfig.getImpl(), null));
        } else {
            Contracts.throwIfFail(createFromConfig(handle, speechConfig.getImpl(), audioConfig.getImpl()));
        }
        recoHandle = new SafeHandle(handle.getValue(), SafeHandleType.SpeakerRecognizer);

        audioInputKeepAlive = audioConfig;
        initialize();
    }


    /**
     * The collection of properties and their values defined for SpeakerRecognizer.
     * @return The collection of properties and their values defined for SpeakerRecognizer.
     */
    public PropertyCollection getProperties() {
        return propertyHandle;
    }

    /**
     * Starts a speaker recognition session to verify the speaker in the verification model.
     * @param model The speaker verification model that specifies the speaker to be verified.
     * @return A task representing the speaker recognition operation. The task returns an instance of SpeakerRecognitionResult.
     */
    public Future<SpeakerRecognitionResult> recognizeOnceAsync(final SpeakerVerificationModel model) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<SpeakerRecognitionResult>() {
            public SpeakerRecognitionResult call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(verifyOnce(recoHandle, model.getImpl(), result));
                return new SpeakerRecognitionResult(result.getValue());
            }
        });
    }

    /**
     * Starts a speaker recognition session to identify the speaker in the identification model.
     * @param model The speaker identification model that specifies the speaker to be identified.
     * @return A task representing the speaker recognition operation. The task returns an instance of SpeakerRecognitionResult.
     */
    public Future<SpeakerRecognitionResult> recognizeOnceAsync(final SpeakerIdentificationModel model) {

        return AsyncThreadService.submit(new java.util.concurrent.Callable<SpeakerRecognitionResult>() {
            public SpeakerRecognitionResult call() {
                IntRef result = new IntRef(0);
                Contracts.throwIfFail(identifyOnce(recoHandle, model.getImpl(), result));
                return new SpeakerRecognitionResult(result.getValue());
            }
        });
    }
    
    /**
     * Dispose of associated resources.
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public void close() {
        if (disposed) {
            return;
        }

        if (propertyHandle != null) {
            propertyHandle.close();
            propertyHandle = null;
        }
        if (recoHandle != null) {
            recoHandle.close();
            recoHandle = null;
        }
        // Deref audioInputKeepAlive by setting to null, this will trigger finalizer if only reference.
        audioInputKeepAlive = null;
                
        AsyncThreadService.shutdown();
        disposed = true;
    }

    /*! \cond INTERNAL */
    /**
     * get the SpeakerRecognizer
     * @return SafeHandle to {@link SpeakerRecognizer}
     */
    public SafeHandle getRecoImpl() {
        return recoHandle;
    }

    /*! \endcond */

    private void initialize() {
        final SpeakerRecognizer _this = this;
        AsyncThreadService.initialize();
        IntRef propHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromRecognizerHandle(_this.recoHandle, propHandle));
        propertyHandle = new PropertyCollection(propHandle);
    }

    private SafeHandle recoHandle = null;
    private AudioConfig audioInputKeepAlive = null;
    private PropertyCollection propertyHandle = null;
    private boolean disposed = false;

    private final native long createFromConfig(IntRef recoHandle, SafeHandle speechConfigHandle, SafeHandle audioConfigHandle);
    private final native long verifyOnce(SafeHandle recoHandle, SafeHandle verificationModel, IntRef resultHandle);
    private final native long identifyOnce(SafeHandle recoHandle, SafeHandle identificationModel, IntRef resultHandle);    
    private final native long getPropertyBagFromRecognizerHandle(SafeHandle recoHandle, IntRef propertyHandle);
}
