//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.List;

import com.microsoft.cognitiveservices.speech.SpeechConfig;
import com.microsoft.cognitiveservices.speech.SpeechRecognitionModel;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.TelemetryManager;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

/**
 * Class that defines embedded (offline) speech configuration.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class EmbeddedSpeechConfig implements AutoCloseable {

    static {
        // Trigger loading of the native libraries.
        try {
            Class.forName(SpeechConfig.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    /**
     * Creates an instance of embedded speech config.
     */
    EmbeddedSpeechConfig(long handleValue) {
        this.config = new SpeechConfig(handleValue);
        TelemetryManager.getSingleton();
    }

    /**
     * Creates an instance of the embedded speech config with a specified offline model path.
     * @param path The folder path to search for offline models.
     * This can be a root path under which several models are located in subfolders,
     * or a direct path to a specific model folder.
     * @return An embedded speech config instance.
     */
    public final static EmbeddedSpeechConfig fromPath(String path) {
        Contracts.throwIfNullOrWhitespace(path, "path");
        IntRef configRef = new IntRef(0);
        Contracts.throwIfFail(createEmbeddedSpeechConfig(configRef));
        EmbeddedSpeechConfig embeddedSpeechConfig = new EmbeddedSpeechConfig(configRef.getValue());
        Contracts.throwIfFail(embeddedSpeechConfigAddPath(embeddedSpeechConfig.config.getImpl(), path));
        return embeddedSpeechConfig;
    }

    /**
     * Creates an instance of the embedded speech config with specified offline model paths.
     * @param paths he folder paths to search for offline models.
     * These can be root paths under which several models are located in subfolders,
     * or direct paths to specific model folders.
     * @return An embedded speech config instance.
     */
    public final static EmbeddedSpeechConfig fromPaths(List<String> paths) {
        if (paths == null || paths.size() == 0) {
            throw new IllegalArgumentException("paths cannot be null or empty");
        }

        IntRef configRef = new IntRef(0);
        Contracts.throwIfFail(createEmbeddedSpeechConfig(configRef));
        EmbeddedSpeechConfig embeddedSpeechConfig = new EmbeddedSpeechConfig(configRef.getValue());
        for (String path : paths) {
            Contracts.throwIfFail(embeddedSpeechConfigAddPath(embeddedSpeechConfig.config.getImpl(), path));
        }
        return embeddedSpeechConfig;
    }

    /**
     * Gets a list of available speech recognition models.
     * @return Speech recognition model info.
     */
    public final List<SpeechRecognitionModel> getSpeechRecognitionModels()
    {
        IntRef numModelsRef = new IntRef(0);
        Contracts.throwIfFail(getNumSpeechRecognitionModels(this.config.getImpl(), numModelsRef));
        int numModels = (int)numModelsRef.getValue();

        List<SpeechRecognitionModel> models = new ArrayList<SpeechRecognitionModel>();

        for (int i = 0; i < numModels; i++) {
            IntRef modelRef = new IntRef(0);
            Contracts.throwIfFail(getSpeechRecognitionModel(this.config.getImpl(), i, modelRef));
            models.add(new SpeechRecognitionModel(modelRef));
        }

        return models;
    }

    /**
     * Sets the model for speech recognition.
     * @param name The model name.
     * @param key The model decryption key.
     */
    public final void setSpeechRecognitionModel(String name, String key) {
        Contracts.throwIfNullOrWhitespace(name, "name");
        config.setProperty(PropertyId.SpeechServiceConnection_RecoModelName, name);

        // Models released in public require a valid decryption key.
        if (key != null && key.length() != 0) {
            config.setProperty(PropertyId.SpeechServiceConnection_RecoModelKey, key);
        }
    }

    /**
     * Gets the model name for speech recognition.
     * @return The recognition model name.
     */
    public final String getSpeechRecognitionModelName() {
        return config.getProperty(PropertyId.SpeechServiceConnection_RecoModelName);
    }

    /**
     * Sets the speech recognition output format.
     * @param value The recognition output format.
     */
    public final void setSpeechRecognitionOutputFormat(OutputFormat value) {
        config.setOutputFormat(value);
    }

    /**
     * Gets the embedded speech recognition output format.
     * @return The recognition output format.
     */
    public final OutputFormat getSpeechRecognitionOutputFormat() {
        return config.getOutputFormat();
    }

    /**
     * Sets the profanity option. This can be used to remove profane words or mask them in output.
     * @param value The profanity option.
     */
    public final void setProfanity(ProfanityOption value) {
        config.setProfanity(value);
    }

    /**
     * Sets the voice for speech synthesis.
     * @param name The voice name for embedded speech synthesis.
     * @param key The decryption key.
     */
    public final void setSpeechSynthesisVoice(String name, String key) {
        Contracts.throwIfNullOrWhitespace(name, "name");
        config.setProperty(PropertyId.SpeechServiceConnection_SynthOfflineVoice, name);
        if (key != null && key.length() != 0) {
            config.setProperty(PropertyId.SpeechServiceConnection_SynthModelKey, key);
        }
    }

    /**
     * Gets the voice name for embedded speech synthesis.
     * @return Returns the embedded speech synthesis voice name.
     */
    public final String getSpeechSynthesisVoiceName() {
        return config.getProperty(PropertyId.SpeechServiceConnection_SynthOfflineVoice);
    }

    /**
     * Sets the embedded speech synthesis output format.
     * @param value The synthesis output format ID (e.g. Riff16Khz16BitMonoPcm).
     */
    public final void setSpeechSynthesisOutputFormat(SpeechSynthesisOutputFormat value) {
        config.setSpeechSynthesisOutputFormat(value);
    }

    /**
     * Gets the embedded speech synthesis output format.
     * @return Returns the synthesis output format.
     */
    public final String getSpeechSynthesisOutputFormat() {
        return config.getSpeechSynthesisOutputFormat();
    }

    /**
     * Gets a list of available speech translation models.
     * @return Speech translation model info.
     */
    public final List<SpeechTranslationModel> getSpeechTranslationModels()
    {
        IntRef numModelsRef = new IntRef(0);
        Contracts.throwIfFail(getNumSpeechTranslationModels(this.config.getImpl(), numModelsRef));
        int numModels = (int)numModelsRef.getValue();

        List<SpeechTranslationModel> models = new ArrayList<SpeechTranslationModel>();

        for (int i = 0; i < numModels; i++) {
            IntRef modelRef = new IntRef(0);
            Contracts.throwIfFail(getSpeechTranslationModel(this.config.getImpl(), i, modelRef));
            models.add(new SpeechTranslationModel(modelRef));
        }

        return models;
    }

    /**
     * Sets the model for speech translation.
     * @param name Model name.
     * @param key Model decryption key.
     */
    public final void setSpeechTranslationModel(String name, String key) {
        Contracts.throwIfNullOrWhitespace(name, "name");
        config.setProperty(PropertyId.SpeechTranslation_ModelName, name);

        // Models released in public require a valid decryption key.
        if (key != null && key.length() != 0) {
            config.setProperty(PropertyId.SpeechTranslation_ModelKey, key);
        }
    }

    /**
     * Gets the model name for speech translation.
     * @return The translation model name.
     */
    public final String getSpeechTranslationModelName() {
        return config.getProperty(PropertyId.SpeechTranslation_ModelName);
    }

    /**
     * Sets a named property as value.
     * @param name the name of the property.
     * @param value the value.
     */
    public void setProperty(String name, String value) {
        config.setProperty(name, value);
    }

    /**
     * Sets the property by propertyId.
     * @param id PropertyId of the property.
     * @param value The value.
     */
    public void setProperty(PropertyId id, String value) {
        config.setProperty(id, value);
    }

    /**
     * Gets a named property as value.
     * @param name the name of the property.
     * @return The value.
     */
    public String getProperty(String name) {
        return config.getProperty(name);
    }

    /**
     * Gets the property by propertyId.
     * @param id PropertyId of the property.
     * @return The value.
     */
    public String getProperty(PropertyId id) {
        return config.getProperty(id);
    }

    /**
     * Dispose of associated resources.
     */
    @Override
    public final void close() {
        if (disposed) {
            return;
        }

        if (config != null)
        {
            config.close();
            config = null;
        }

        disposed = true;
    }

    /*! \cond INTERNAL */
    /**
     * Returns a internal handle to SpeechConfig implementation.
     * @return The implementation handle.
     */
    public SafeHandle getImpl() {
        Contracts.throwIfNull(config, "config");
        return config.getImpl();
    }
    /*! \endcond */

    SpeechConfig config = null;

    private final static native long createEmbeddedSpeechConfig(IntRef configHandle);
    private final static native long embeddedSpeechConfigAddPath(SafeHandle configHandle, String path);
    private final native long getNumSpeechRecognitionModels(SafeHandle configHandle, IntRef numModelsRef);
    private final native long getSpeechRecognitionModel(SafeHandle configHandle, int index, IntRef modelRef);
    private final native long getNumSpeechTranslationModels(SafeHandle configHandle, IntRef numModelsRef);
    private final native long getSpeechTranslationModel(SafeHandle configHandle, int index, IntRef modelRef);

    private boolean disposed = false;
}
